{
	"translatorID": "df9dca51-1f66-4dc6-a3e1-09b4c416db23",
	"label": "NYPL Menus",
	"creator": "Abe Jellinek",
	"target": "^https?://menus\\.nypl\\.org/",
	"minVersion": "3.0",
	"maxVersion": "",
	"priority": 100,
	"inRepository": true,
	"translatorType": 4,
	"browserSupport": "gcsibv",
	"lastUpdated": "2021-06-23 00:03:16"
}

/*
	***** BEGIN LICENSE BLOCK *****

	Copyright © 2021 Abe Jellinek
	
	This file is part of Zotero.

	Zotero is free software: you can redistribute it and/or modify
	it under the terms of the GNU Affero General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Zotero is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU Affero General Public License for more details.

	You should have received a copy of the GNU Affero General Public License
	along with Zotero. If not, see <http://www.gnu.org/licenses/>.

	***** END LICENSE BLOCK *****
*/


function detectWeb(doc, url) {
	if (/\/menu(s|_pages)\/\d+/.test(url)) {
		return "document";
	}
	else if (getSearchResults(doc, true)) {
		return "multiple";
	}
	return false;
}

function getSearchResults(doc, checkOnly) {
	var items = {};
	var found = false;
	var rows = doc.querySelectorAll('.item .menu');
	for (let row of rows) {
		let href = attr(row, 'a[href*="/menus/"]', 'href');
		let title = ZU.trimInternal(text(row, 'a:not(.thumbportrait)') + ' '
			+ (text(row, 'div').match(/([0-9]{4})/) || [''])[0]);
		if (!href || !title) continue;
		if (checkOnly) return true;
		found = true;
		items[href] = title;
	}
	return found ? items : false;
}

function doWeb(doc, url) {
	if (detectWeb(doc, url) == "multiple") {
		Zotero.selectItems(getSearchResults(doc, false), function (items) {
			if (items) ZU.processDocuments(Object.keys(items), scrape);
		});
	}
	else if (url.includes('/menu_pages/')) {
		ZU.processDocuments(attr(doc, '#carousel li:first-child a', 'href'), scrape);
	}
	else {
		scrape(doc, url);
	}
}

function scrape(doc, _url) {
	let item = new Zotero.Item('document');
	item.title = text(doc, 'div.modes') + ' menu';
	item.libraryCatalog = 'New York Public Library Menu Collection';
	item.numPages = '' + doc.querySelector('.pages').childElementCount;
	item.url = attr(doc, '.pages a', 'href').replace(/\/explore$/, '');
	if (item.url.startsWith('/')) {
		// only one case (/menu_pages/12345) to deal with here.
		// no HTTPS.
		item.url = 'http://menus.nypl.org' + item.url;
	}
	item.extra = 'genre: Restaurant menu\n';
	
	for (let p of doc.querySelectorAll('.metadata p')) {
		let text = ZU.trimInternal(p.innerText);
		if (text.startsWith('Location')) {
			text = text.substring('Location'.length).trim();
			item.extra += `Restaurant Location: ${text}\n`;
		}
		else if (text.startsWith('Date')) {
			text = text.substring('Date'.length);
			item.date = ZU.strToISO(text);
		}
		else if (text.startsWith('Place')) {
			text = text.substring('Place'.length).trim();
			item.extra += `Place: ${text}\n`;
		}
		else if (text.startsWith('Physical description')) {
			text = text.substring('Physical description'.length).trim();
			item.extra += `Physical Description: ${text}\n`;
		}
		else if (text.startsWith('Call number')) {
			text = text.substring('Call number'.length);
			item.callNumber = text;
		}
		else if (text.startsWith('Notes')) {
			text = text.substring('Notes'.length);
			item.abstractNote = text;
		}
	}
	
	let note = '<h1>Dishes</h1><ul>';
	for (let row of doc.querySelectorAll('.dishtable > tbody > tr')) {
		let name = text(row, '.name');
		let price = text(row, '.price');
		if (price == '-') {
			note += `<li>${name}</li>`;
		}
		else {
			note += `<li>${name} (price: ${price})</li>`;
		}
	}
	note += '</ul>';
	item.notes.push({ note });
	
	item.complete();
}

/** BEGIN TEST CASES **/
var testCases = [
	{
		"type": "web",
		"url": "http://menus.nypl.org/menus/26680",
		"items": [
			{
				"itemType": "document",
				"title": "Zum Durnbrau menu",
				"creators": [],
				"date": "2005-01-14",
				"abstractNote": "Dieter Zander Collection; includes three paper inserts.",
				"callNumber": "Zander 324",
				"extra": "genre: Restaurant menu\nRestaurant Location: Zum Durnbrau\nPlace: Zum Durnbrau\nPhysical Description: Tri-fold; 11.75 x 16.5 inches folded",
				"libraryCatalog": "New York Public Library Menu Collection",
				"url": "http://menus.nypl.org/menu_pages/46080",
				"attachments": [],
				"tags": [],
				"notes": [
					{
						"note": "<h1>Dishes</h1><ul><li>Gebrannte Grieβsupp'n  3.90 (price: 3.9)</li><li>Broccolirahmsupp'n  4.10 (price: 4.1)</li><li>Gebrannte Grießsupp'n, Seelachsfilet Gebacken, Remouladensoße, Kartoffel Gurkensalat (price: 9.7)</li><li>Seelachsfilet Gebacken, Remouladensosse, Kartoffel Gurkensalat (price: 9.7)</li><li>Saure Brez'n Knödel, Dünn Aufgeschnitten, Mit Regensburger Wurst (price: 7.2)</li><li>Heringssalat Mit Zwiebelringen Und Hausbrot  8.20 (price: 8.2)</li><li>Blattsalat In Kräuterrahmdressing Mit In Butter Gebratenen Austernpilzen (price: 6.4- 0.0)</li><li>Schweinskopf   Sülze, Hausgemacht Mit Röstkartoffeln  9.20 / Klein  6.40 (price: 6.4- 0.0)</li><li>2  Ochsenaugen Auf Rahmspinat, Dazu Salzkartoffeln (price: 6.6)</li><li>Schweinsleber Geschnetzelt, Sauer, Mit Röstkartoffeln Und Grünem Salat  8.70 / Klein  6.10 (price: 6.1- 0.0)</li><li>Käsespätzle Hausgemacht, Mit Röstzwiebeln Und Gemischtem Salatteller  8.70 / Klein  6.10 (price: 6.1- 0.0)</li><li>Tagesgemüse Blumenkohl Mit Schinken Und Käse überbacken, Salzkartoffeln (price: 9.2)</li><li>Kassler Vom Rost, Mit Gemischtem Kartoffel   Gurkensalat  9.20 (price: 9.2)</li><li>Hühnerbrüstchen Goldgelb Gebacken Mit Remouladensoße Und Gemischtem Salat  9.70 (price: 9.7)</li><li>Geschmorte Ochsenwade In Zwiebelsoße Mit Brezenknödel Und Grünem Salat  10.20 / Klein  7.10 (price: 7.1- 0.0)</li><li>Schweinsbraten Mit Kruste, Dazu Kartoffelknödel Und Speckkrautsalat  10.20 / Klein  7.10 (price: 7.1- 0.0)</li><li>1/2 Schweinshax'n Knusprig Gebraten, Mit Kartoffelknödel Und Speckkrautsalat  12.30 (price: 12.3)</li><li>Spanferkel Knusprig, In Dunkelbiersoße Mit Kartoffelknödel Und Speckkrautsalat  16.40 / Klein  11.50 (price: 11.5- 0.0)</li><li>Schellfisch Gekocht, In Senfrahmsoße Mit Petersilienkartoffeln Und Blattsalat  10.20 (price: 10.2)</li><li>Dampfnudel Mit Heißer Vanillesoße  5.10 (price: 5.1)</li><li>Castell Castell [Ww) Silvaner Qb A   Qualitätswein Trocken Duft Nach Pfirsich Und Banane, Jung, Frisch, Handlese, Cremig Und Weich (price: 0.25- 0.0)</li><li>Spatenbräu Hell, Feinstes Münchner Lagerbier  0,5 L  3,20 (price: 3.2)</li><li>Spatenbräu Pils  25 L  1,90 / 0,3 L  3,20 (price: 1.9- 0.0)</li><li>Spatenbräu Dunkles Bier  0,5 L  3,20 (price: 3.2)</li><li>Schloss Castell [Ww) Bacchus Qb A   Qualitätswein  0,25  5,00 (price: 5.0)</li><li>Schnitt (Schaumhalbe), Hell Oder Dunkel Lagerbier  0,25 L  1,90 / 0,4 L  2,80 (price: 1.9- 0.0)</li><li>Franziskaner Hefeweissbier Hell  O,5 L  3,50 (price: 3.5)</li><li>Radler (Lagerbier Mit Brauselimonade)  0,3 L  2,30 / 0,5 L  3.20 (price: 2.3- 0.0)</li><li>Grüner Veltliner   Vom Urgestein   Trocken [Ww) Weingut Franz Türk  0,25  7,50 / 0,75  22,00 (price: 7.5- 0.0)</li><li>Russen (Hefeweissbier Mit Brauselimonade)  0,5 L  3,50 (price: 3.5)</li><li>Pfälzer Landwein   Rheinpfalz, Landau (Ww), Deutscher Tafelwein   Steiner'scher Keller  0,35  5,00 (price: 5.0)</li><li>Franziskaner Hefeweissbier Dunkel  0,5 L  3,50 (price: 3.5)</li><li>Spaten Alkoholfrei, Ein Herbfrisches Schankbier  0,5 L  3,20 (price: 3.2)</li><li>Light Bier, Das Kalorienarme Mit Weniger Alkohol  0,5 L  3,70 (price: 3.2)</li><li>Triumphator, Dunkles Starkbier  0,5 L  3,70 (price: 3.7)</li><li>Spaten Diätpils  0,33 L  3,20 (price: 3.2)</li><li>Castell (Rose), Rotling Qb A   Qualitätswein Trocken  0,25  5,00 (price: 5.0)</li><li>Hefeweizen Leicht  0,5 L  3,50 (price: 3.5)</li><li>Reichsgraf Von Kesselstatt (Rw), Riesling Feinherb   Qualitätswein Mosel Saar Ruwer  0,25  5,50 (price: 5.5)</li><li>Gerolsteiner Original Mineralwasser  0,25 L  2,20 (price: 2.2)</li><li>Staatlich Fachingen Mineralwasser  0,5 L  3,20 /   0,25 L  2,40 (price: 2.4- 0.0)</li><li>Zweigelt   Weingut Markowitsch (Rw), Landwein   Qualitätswein Aus Dem Carnuntum  0,25  5,50 (price: 5.5)</li><li>Libella Zitronenlimonade  0,5 L  3,40 (price: 3.4)</li><li>Pepsi Cola (Coffeinhaltige Limonade)  0,33 L  2,50 (price: 2.5)</li><li>Mirinda  0,33 L  2,50 (price: 2.5)</li><li>Spezi   Schwip Schwap (Coffeinhaltige Limonade)  0,5 L  3,50 (price: 3.5)</li><li>Apfelsaft  0,2 L  2,40 (price: 2.4)</li><li>Orangensaft  0,2 L  2,70 (price: 2.7)</li><li>Trollinger/Lemberger (Rw), Haberschlachter Heuchelberg, Qb A Schlosskellerei Affaltrach, Würtemberg  0,35  6,00 (price: 6.0)</li><li>Schwarzer Johannisbeernektar  0,2 L  2,70 (price: 2.7)</li><li>Tomatensaft  0,2 L  3,30 (price: 3.3)</li><li>Apfelschorle  0,25 L  2,10 (price: 2.1)</li><li>Apfelschorle  0,5 L  3,70 (price: 3.7)</li><li>Tasse Kaffee  2,30 (price: 2.3)</li><li>Haferl Kaffee  3,70 (price: 3.7)</li><li>Obstler Von Stettner  38 % Vol  2 Cl  2,50 (price: 2.5)</li><li>Espresso  2,00 (price: 2.0)</li><li>Cappuccino  2,70 (price: 2.7)</li><li>Altbayerischer Kartoffelschnaps  40 % Vol  2 Cl  2,50 (price: 2.5)</li><li>Heisse Schokolade Mit Schlagrahm  3,50 (price: 3.5)</li><li>Milch Kaffee  3,70 (price: 3.7)</li><li>Bärwurz Wurzelschnaps  40 % Vol  2 Cl  2,50 (price: 2.5)</li><li>Glas Tee  2,30 (price: 2.3)</li><li>Steinhäger, Dornkaat  28 % Vol  2 Cl  2,50 (price: 2.5)</li><li>Glühwein  4,50 (price: 4.5)</li><li>Grog Von 4 Cl Pott Rum  4,90 (price: 4.9)</li><li>Gebirgsenzian  40 % Vol  2 Cl  2,70 (price: 2.7)</li><li>Kaffee Hag (price: 2.3)</li><li>Zwetschgenwasser, Himbeergeist, Williamsbrine Oder Kirschwasser Je  42 % Vol  2 Cl  3,00 (price: 3.0)</li><li>Williamsbirne (price: 3.0)</li><li>Jägermeister  35 % Vol  2 Cl  2,70 (price: 2.7)</li><li>Underberg  44 % Vol  2 Cl  2,90 (price: 2.9)</li><li>Amaretto Di Saronne  2cl  3,00 (price: 3.0)</li><li>Fernet Branca  (42 %) Oder Fernet Menta (40%)  2 Cl  2,90 (price: 2.9)</li><li>Ramazzotti (30 %) Avernar (40 %)  2 Cl  3,20 (price: 3.2)</li><li>Hennessy Cognac Vsop  2 Cl  3,70 (price: 3.7)</li><li>Asbach Uralt  38 % Vol  2 Cl  3,20 (price: 3.2)</li><li>Drambuie  2 Cl  3,50 (price: 3.5)</li><li>Hennessy Cognac Vsop  40 % Vol  2 Cl  3,70 (price: 3.7)</li><li>Calvados  42 % Vol  2 Cl  4,30 (price: 4.3)</li><li>La Vieille Prune, Ein Alter Pflaumenbrand  42 % Vol  2 Cl  4,70 (price: 4.7)</li><li>Hennessy Cognac Xo, 50 Jahre Alt  40 % Vol  2 Cl  9,80 (price: 9.8)</li><li>Jubiläums Aquavit  42 % Vol  2 Cl  3,20 (price: 3.2)</li><li>Malteser Aquavit  40 % Vol  2 Cl  3,30 (price: 3.3)</li><li>Linie Aquavit  41,5 % Vol  2 Cl  4,20 (price: 4.2)</li><li>Morand Williamine  Oder Kirsch   43 % Vol  2 Cl  5,40 (price: 5.4)</li><li>Ziegler Sauerkirsche  43 % Vol  2 Cl  5,60 (price: 5.6)</li><li>Ziegler, Nr 1, Wildkirsche  43 % Vol  2 Cl  9,70 (price: 9.7)</li><li>Veith Adams Bierschnaps  42 % Vol  2 Cl  5,60 (price: 5.6)</li><li>Grappa Di Prosecco  42 % Vol  2 Cl  5,60 (price: 5.6)</li><li>Feigling   Wodka Feige  2 Cl0  3,70 (price: 3.7)</li><li>Campari*  4 Cl  3,70 (price: 3.7)</li><li>Sherry, Halbtrocken Oder Trocken  5 Cl  3,70 (price: 3.7)</li><li>Lantenhammer Schlehengeist, Waldhimheergeist, Mirabellenbrand, Williamsbrand, Sauerkirschbrand  42 % Vol  2 Cl  4,70 (price: 4.7)</li><li>Pflaumenwein  5 Cl  4,30 (price: 4.3)</li><li>Aperol (Rhabarberbitter)  2 Cl  3,00 (price: 3.0)</li><li>Lantenhammer Obstler  40 % Vol  2 Cl  3,70 (price: 3.7)</li><li>Sauers Lüngerl Mit Semmelknödel  8,60 (price: 8.6)</li><li>Kälberne Milzwurst, Gebacken Mit Kartoffel Gurkensalat (Oder Aus'm Wurzelsud, Sauer, Mit Friscem Meerrettich)  10,20 (price: 10.2)</li><li>Abgeschmolzene Maultaschen In Zwiebelsoße, Mit Grünem Salat  10,50 (price: 10.5)</li><li>Kälberfüße, Ausgelöst Und Paniert, In Butter Gebacken, Mit Remouladensoße, Und Gemischtem Salatteller (Oder Kälberfüsse Aus'm Wurzelsud, Sauer, Mit Frischem Meerrettich)  10,50 (price: 10.5)</li><li>Ausgelöster Kalbskopf, Gebacken Mit Remouladensoße Und Gemischtem Kartoffel Gurkensalat (Oder Ausgelöster Kalbskopf Aus'm Wurzelsud, Sauer, Mit Frischem Meerrettich)  11,50 (price: 11.5)</li><li>Ochsenfleisch, Paniert, In Butter Gebacken, Mit Remouladensoße Und Kartoffel Gurkensalat  12,20 (price: 12.2)</li><li>Ochsenfleisch Mit Ei Und Jungen Zwiebeln In Butter Abgeröstet, Dazu Salzkartoffeln Und Blattsalat  12,20 (price: 12.2)</li><li>Surhax'n (Eisbein) Auf Sauerkraut Mit Röstkartoffeln  12,20 (price: 12.2)</li><li>Kalbsbries Natur, In Butter Gebraten, Mit Champignon Kräuterrahmsosse Dazu Petersilienkartoffeln  17,80 (price: 17.8)</li><li>Kalbsbries Paniert, In Butter Gebacken, Mit Remouladensoße Und Kartoffel Gurkensalat  17,80 (price: 17.8)</li><li>Niederbayerische Mast Gans Aus'm Bratrohr Mit Apfelblaukraut Und Handgeriebenen Kartoffelknödeln</li><li>Niederbayerische Mast Ente Aus'm Bratrohr Mit Apfelblaukraut Und Handgeriebenen Kartoffelknödeln</li><li>Schweinekotelett Paniert In Butter Gebacken Mit Gemischtem Kartoffel Gurkensalat  12,00 (price: 12.0)</li><li>Schweinkotelett Natur, In Krautersoße Mit Röstkartoffeln Und Gemischten Salat  12,20 (price: 12.2)</li><li>Schweineschnitzel Gebacken, Mit Röstkartoffeln Und Gemischtem Salatteller  12,00 (price: 12.0)</li><li>Schweinefilet Vom Grill, Mit Champignons, Petersiliekartoffeln Und Frischem Marktgemüse  15,30 (price: 15.3)</li><li>Wiener Schnitzel Vom Kalb, In Butter Gebacken, Mit Röstkartoffeln Und Gemischtem Salatteller  16,30 (price: 16.3)</li><li>Zwiebelrostbraten (Aus Der Lende) Mit Röstkartoffeln Und Grünen Bohnen  16,30 (price: 16.3)</li><li>Filetsteak Vom Rind, In Gestoßenem Pfeffer Gebraten, In Leichter Cognac Rahmsoße, Mit Butterreis Und Frischem Marktgemüse  21,40 (price: 21.4)</li><li>Schlachtschüssel Blut  Und Leberwurst Mit Wellfleisch Auf Sauerkraut, Dazu Kartoffelpüree.</li><li>Original Miesbacher, Delikatesskase   Auch Sauer   Mit Zwiebelringen  4,30 (price: 4.3)</li><li>Allgäuer Emmentaler Käse, Aufgeschnitten  6,60 (price: 6.6)</li><li>Obatzter Käse, Garniert Mit Zwiebelringen  7,10 (price: 7.1)</li><li>Echter Gorgonzola   Auch Sauer   Mit Zwiebelringen  9,70 (price: 9.7)</li><li>Bayerischer Käseteller  12,20 (price: 12.2)</li><li>Kleiner Pfannkuchen Mit Marmelade Gefüllt  3,20 (price: 3.2)</li><li>Apfelstrudel Mit Heißer Vanillesoße  4,70 (price: 4.7)</li><li>Gemischtes Eis Mit Schlagrahm  4,70 (price: 4.7)</li><li>Bayerische Creme Mit Schlagrahm Und Fruchtsoße  5,10 (price: 5.1)</li><li>Eisbecher Mit Amarenakirschen Und Eingelegten Pflaumen, Schlagrahm  7,60 (price: 7.6)</li><li>Semmelschmarrn Mit Zwetschgenkompott  9,10 (price: 9.1)</li><li>Kaiserschmarrn Mit Mandeln, Rosinen Und Apfelmus  10,20 (price: 10.2)</li><li>Bayerische Brotsuppe, Abgeschmolzen, Mit Gerösteten Zwiebeln  4,50 (price: 4.5)</li><li>Kräftige Kesselbrühe, Mit Hausgemachtem Leberknödel  4,50 (price: 4.5)</li><li>Münchner Kartoffelsuppe, Mit Lauchstreifen Und Räucherspeck  4,90 (price: 4.9)</li><li>Omelette Natur Mit Grünem Salat  6,40 (price: 6.4)</li><li>Rühreier Mit Schinken Und Grünem Salat  7,10 (price: 7.1)</li><li>Omelette Mit Schinken Oder Champignons Oder Käse Und Grünem Salat  8,60 (price: 8.6)</li><li>Speckpfannenkuchen Mit Gemischtem Salatteller  8,90 (price: 8.9)</li><li>Münchner Weisswurst*, Ab 2 Stück (price: 1.9)</li><li>Wiener Würstchen (price: 3.8)</li><li>Debreciner Würstchen (price: 4.0)</li><li>Münchner Stadtwurst Vom Rost Auf Sauerkraut (price: 6.6)</li><li>Münchner Stadtwurst Heissgelegt Auf Sauerkraut (price: 6.6)</li><li>Münchner Schweinsbratwürstl Mit Kraut (price: 6.1)</li><li>Gschwollne Würstl Mit Kartoffelsalat (price: 6.6)</li><li>Orig. Nürnberger Rostbratwürstl Mit Kraut (price: 4.9- 0.0)</li><li>4 Stuck Orig.  Nurnberger Rostabratwurstl* Mit Kraut (price: 4.9)</li><li>8 Stuck Saure Zipfel* Im Wurzelsud Serviert (price: 9.1)</li><li>4 Stuck Saure Zipfel Im Wurselsud Serviert (price: 4.9)</li><li>Portion Bayerischer Leberkäs, Ofengebacken, Warm Serviert Mit Hausmachersenf (price: 6.4)</li><li>Leberkäs Kalt Mit Essiggurke Dazu Hausgemachter, Süsser Senf (price: 6.4)</li><li>Wurstsalat Aus Regensburgern Mit Zwiebelringen (price: 6.6)</li><li>Bayerischer Bauernpresssack Mit Zwiebelringen Schwarz Und Weiss (price: 6.6)</li><li>Bayerischer Bauernpresssack In Essig Und öl (price: 6.6)</li><li>Ochsenmaulsalat, Süss Sauer Angemacht Mit Zwiebeln (price: 7.6)</li><li>Knöcherlsülze Hausgemacht Mit Röstkartoffeln (price: 7.6)</li><li>Bayerischer Lumpensalat Aus Rindfleisch, Wurst, Käse, Tomate Und Zwiebeln (price: 7.6)</li><li>Kalter Braten, Dünn Geschnitten Mit Gemischtem Kartoffel Gurkensalat (price: 10.2)</li><li>Münchner Brotzeitbrettl Mit Hausbrot  12,50 (price: 12.5)</li><li>Orig. Niederbayerisches Bauerngeräuchertes Mit Apfelkren Und Hausbrot  12,70 (price: 12.7)</li><li>Bauernspeck, Luftgetrocknet, Dünn Aufgeschnitten  13,20 (price: 13.2)</li><li>Ochsenfleisch, Dünn Aufgeschnitten, Suüß Sauer Angemacht, Mit Zwiebel  Und Gurkenstreifen, Röstkartoffeln  12,20 (price: 12.2)</li><li>Roastbeef, Rosa Gebraten Mit Remouladensoße Und Röstkartoffeln  16,80 (price: 16.8)</li><li>Tatar Garniert, Mit Butter Und Schwarzbrot  15,30 (price: 15.3)</li><li>Gemischte Salatplatte Aus Frischen Marktsalaten Mit Gekochtem Schinken  8,60 (price: 8.6)</li><li>Milde Matjesfilets In Apfel Sahnesoße Mit Butterkartoffeln  10,20 (price: 10.2)</li><li>Bierradi Mit Schnittlauchbrot  Nach Saison!</li><li>Ochsenmaul Salat Suss Sauer Angemacht (price: 7.7)</li><li>Wade Gekocht, Dunn Aufgeschnitten, Suss Sauer Auf Blattsalat (price: 8.7)</li><li>Sulze, Vom Gekochten Tafelspitz Mit Gemuse, Dazu Rostkartoffeln (price: 11.2)</li><li>Tages Rahmgemüse</li><li>Gurkensalat</li><li>Blattsalat</li><li>Röstkartoffeln</li><li>Saures Kartoffelgemüse</li><li>Salzkartoffeln</li><li>Meerrettich Rahmsosse</li><li>Wurzelsud</li><li>Apfelmeerrettich</li><li>Schnittlauch Rahmsosse</li><li>Kutteln, In Brauner Sosse, Sauer Gekocht (price: 7.7)</li><li>Zunge, Gepokelt Und Gekocht, Zart Und Schmackhaft (price: 14.5)</li><li>Kronfleisch (Das Zwerchfell) Leicht Grobfaserig, Aber Mit Biss (price: 13.9)</li><li>Kopfbackerl  Butterweich, Leicht Durchzogen (price: 14.5)</li><li>Tafelspitz (price: 17.9)</li><li>Brust Kern (price: 14.9)</li><li>Burgermeister Stuck (price: 19.9)</li><li>Fledermaus (price: 19.9)</li><li>Lende (price: 18.9)</li><li>Filet (price: 23.9)</li><li>Gemischtes Gekochtes, Drei Sorten Vom Ochs Mit Zunge In Einem Topf (price: 18.9)</li><li>Pfannkuchensupp'n  3.90 (price: 3.9)</li><li>Fenchelrahmsupp'n  4.10 (price: 4.1)</li><li>Camembert Gebacken Mit Preiselbeeren Und Rosti (price: 5.1)</li><li>Saure Brez'n Knodel, Dunn Aufgeschnitten, Mit Regensburger Wurst (price: 7.2)</li><li>Blattsalat In Krauterrahmdressing Mit Truthahnstreifen (price: 8.2)</li><li>Schweinskopf   Sulze, Hausgemacht Mit Rostkartoffeln</li><li>3 Reiberdatschi Auf Sauerkraut Oder Mit Apfelmus (price: 7.2)</li><li>Hackbraten Mit Schwammerlrahmsosse, Butternudeln Und Grunem Salat (price: 8.2)</li><li>Schweinsleber Geschnetzelt, Sauer, Mit Rostkartoffeln Und Grunem Salat (price: 8.7)</li><li>Kasespatzle Hausgemacht, Mit Rostzwiebeln Und Gemischtem Salatteller (price: 8.7)</li><li>Zucchini Paprikagemuse Mit Schinken Und Kase Uberbacken, Kartoffeln (price: 9.2)</li><li>Huhnerbruschtchen Goldgelb Gebacken Mit Remouladensosse Und Gemischtem Salat (price: 9.7)</li><li>Geschmorte Ochsenwade In Zwiebelsosse Mit Brezenknodel Und Grunem Salat (price: 10.2)</li><li>Schweinsbraten Mit Kruste, Dazu Kartoffelknodel Und Speckkraustsalat (price: 10.2)</li><li>1/2 Schweinshax'n Knusprig Gebraten, Mit Kartoffelknodel Und Speckkrautsakat (price: 12.3)</li><li>Spanferkel Knusprig, In Dunkelbiersosse Mit Kartoffelknodel Und Speckkrautsalat (price: 16.4)</li><li>Cordon Bleu Vom Kalb, In Butter Gebraten Mit Rostkartoffeln Und Gemischtem Salatteller (price: 17.4)</li><li>Heilbutt Vom Grill, Mit Krauterbutter, Kartoffeln Und Salat (price: 14.3)</li><li>Apfelkucherl In Bierteig Gebacken, Mit Zimtzucker Und Vanilleeis (price: 5.1)</li></ul>"
					}
				],
				"seeAlso": []
			}
		]
	},
	{
		"type": "web",
		"url": "http://menus.nypl.org/menus/28929/explore",
		"items": [
			{
				"itemType": "document",
				"title": "The Famous Anchor Sea Food House menu",
				"creators": [],
				"date": "1962-07-29",
				"abstractNote": "German is secondary language",
				"callNumber": "1962-0040_wotm",
				"extra": "genre: Restaurant menu\nRestaurant Location: The Famous Anchor Sea Food House\nPhysical Description: 33.5x25.5cm folded; 33.5x51cm open",
				"libraryCatalog": "New York Public Library Menu Collection",
				"url": "http://menus.nypl.org/menu_pages/54352",
				"attachments": [],
				"tags": [],
				"notes": [
					{
						"note": "<h1>Dishes</h1><ul><li>Fresh Fruit In Season</li><li>Manhattan Clam Chowder (price: 0.35- 25.0)</li><li>Tomato Juice (price: 0.25)</li><li>Jumbo Shrimp Cocktail (price: 1.2)</li><li>New England Fish Chowder (price: 0.35- 0.5)</li><li>Fresh Fruit Cup (price: 0.45)</li><li>Crab Lump Cocktail (price: 1.5)</li><li>Fresh Seafood Cocktail (price: 2.25)</li><li>Tomato Juice (price: 0.25)</li><li>Clam Broth (price: 0.25- 50.0)</li><li>Lobster Cocktail (Fresh Picked) (price: 1.95)</li><li>Filet Of Marinated Herring, Cream Sauce (price: 0.55)</li><li>Cherrystone Clams (6) (price: 0.75)</li><li>Clams A La Casino (price: 2.25)</li><li>Fried Clams, Tartare (price: 1.95)</li><li>Oysters: Blue Points (6) On The Half Shell (price: 0.95)</li><li>Clam Stew In Milk (price: 1.5)</li><li>Clam Stew, Half And Half (price: 1.85)</li><li>Oysters: Cape Cods (6) On The Half Shell</li><li>Broiled Oysters (Crumb) (price: 2.25)</li><li>Oysters A La Casino (price: 3.25)</li><li>Oyster Stew: Half And Half (price: 2.35)</li><li>Oyster Stew: Milk (price: 2.1)</li><li>Oyster Fry, Sauce Tartare (price: 2.1)</li><li>Basket Steamed Soft Main Clams, Broth, Drawn Butter (price: 1.95)</li><li>Oyster Fry With Bacon, Sauce Tartare (price: 2.1)</li><li>Sea Scallops Saute (price: 2.45)</li><li>Sea Scallops Newburg (price: 2.45)</li><li>Peconic Deep Sea Scallops, Fried (price: 2.15)</li><li>Peconic Deep Sea Scallops, Broiled (price: 2.45)</li><li>Long Island Bay Scallops, Fried (price: 2.5)</li><li>Long Island Bay Scallops, Broiled (price: 2.75)</li><li>Long Island Bay Scallops Saute (price: 2.75)</li><li>Sea Scallops Au Gratin (price: 2.75)</li><li>Jumbo Shrimp Cocktail (price: 1.2)</li><li>Curried Shrimps, With Steamed Rice, Chutney (price: 2.75)</li><li>Fried, Florida Shrimps (price: 2.45)</li><li>Shrimps A La Newburg (price: 2.75)</li><li>Shrimps Creole (price: 2.75)</li><li>King Sized Jumbo Shrimp Salad (price: 2.65)</li><li>Shrimps Saute In Butter (price: 2.75)</li><li>Shrimps Au Gratin (price: 2.75)</li><li>Cold Half Lobster Platter (price: 2.25)</li><li>Lobster Thermidor (price: 4.25)</li><li>Lobster Saute (price: 3.45)</li><li>Lobster A La Newburg (price: 3.45)</li><li>Lobster Stew In Milk (price: 2.25)</li><li>Lobster Stew In Half And Half (price: 2.65)</li><li>Lobster Au Gratin (price: 3.25)</li><li>Lobster A L'americaine (price: 4.95)</li><li>Bouillabaise Marseillaise (price: 4.95)</li><li>Broiled Jumbo Deviled Crab (1) (price: 1.75)</li><li>Broiled Jumbo Deviled Crab (Pair) (price: 2.45)</li><li>Broiled South African Lobster Tail, Drawn Butter (price: 3.75)</li><li>Crabs A La Newburg (price: 2.95)</li><li>Crab Lump Salad (price: 2.65)</li><li>Crab Meat Au Gratin (price: 2.95)</li><li>Crab Flakes, Dewey (price: 2.75)</li><li>Crabmeat Saute (price: 2.95)</li><li>Soft Shell Crabs    Broiled    Fried</li><li>Broiled Mushrooms (price: 0.75)</li><li>Grilled Tomato (price: 0.35)</li><li>French Fried Onion Rings (price: 0.45)</li><li>Green Vegetable (In Season)</li><li>Shrimps, Scallops And Clams, Fried, Tartare (price: 2.35)</li><li>Shrimps, Scallops And Clams, Saute In Pure Butter (price: 2.5)</li><li>Lobster, Shrimps And Crab Meat Stew In Milk (price: 1.75)</li><li>Crabmeat Salad (price: 2.65)</li><li>Lobster, Shrimps And Crab Meat Stew Au Gratin (price: 3.25)</li><li>Shrimp Salad (price: 2.65)</li><li>Lobster (Fresh) Salad (price: 3.5)</li><li>Lobster, Shrimps And Crab Meat Saute In Butter (price: 3.25)</li><li>Combination Salad (Lobster, Shrimp And Crab Meat) (price: 3.25)</li><li>Lobster, Shrimps And Crab Meat A La Newburg (price: 3.25)</li><li>Chef's Salad Bowl, Anchor's Special Dressing (price: 0.25)</li><li>Hearts Of Lettuce Salad (price: 0.35)</li><li>Lettuce And Tomato Salad (price: 0.5)</li><li>Sliced Tomato Salad (price: 0.35)</li><li>Melon</li><li>Cup Of Chowder</li><li>Fresh Fruit Cup</li><li>Lump Crabmeat, Shrimp And Rock Lobster Au Gratin   En Casserole (price: 3.25)</li><li>Broiled Boneless Shad And Shad Roe, Parsley Butter   Crisp Bacon (price: 3.25)</li><li>Cold 1/2 Lobster (price: 2.25)</li><li>Broiled Eastern Halibut Steak Parsley Butter (price: 2.75)</li><li>Selected Large Sea Bass Split And Broiled (price: 2.75)</li><li>Broiled Jumbo Deviled Crabs Prepared To Order (price: 2.75)</li><li>Broiled Large Long Island Flounder Lemon Butter (price: 2.75)</li><li>Broiled Fresh Calf's Liver, Crisp Bacon (price: 2.45)</li><li>Fresh Calf's Liver, Smothered Onions (price: 2.45)</li><li>Tea</li><li>Coffee</li><li>Individual Serving Milk</li><li>Two Vegetables</li><li>Broiled Large Deep Sea Scallops (price: 2.75)</li><li>Fried Maryland Soft Shell Crabs (price: 2.95)</li><li>Broiled Florida Pompano (price: 2.75)</li><li>Seas Kebab   A Tempting Combination Of Sea Scallops, Jumbo Florida Shrmps, Maine Lobster Meat, Crisp Bacon (Broiled On Skewer) Served On A Bed Of Saffron Rice.   Served With Choice Of Two   Vegetable Du Jour, French Fried Potatoes, Chef's Salad Bowl (price: 2.95)</li><li>Tomato Juice</li><li>Broiled Fresh Long Island Flounder (price: 1.95)</li><li>Apple Juice</li><li>Grapefruit Juice</li><li>Fruit Cocktail</li><li>Golden Fried Filet Of Native Lemon Sole, Sauce Tartar (price: 2.25)</li><li>Cherrystone Clams</li><li>Shrimp Cocktail</li><li>Large Selected Sea Bass, Split And Broiled (price: 2.45)</li><li>New England Fish Chowder (Cup)</li><li>Broiled Filet Of Lake Erie Whitefish (price: 2.65)</li><li>Manhattan Clam Chowder (Cup)</li><li>Clam Broth (Cup)</li><li>Broiled Filet Of Fresh Long Island Fluke (price: 2.45)</li><li>Steamed Clams (When Available)</li><li>Broiled Fresh Striped Bass, Parlsey And Butter (price: 2.45)</li><li>Broiled Chinook Salmon Steak With Butter (price: 2.75)</li><li>Poached Chinook Salmon Steak With Butter (price: 2.75)</li><li>Broiled Deviled Crab</li><li>Thickly Cut Swordfish Steak, Broiled (price: 2.45)</li><li>Thickly Cut Swordfish Steak, Saute (price: 2.45)</li><li>Broiled Filet Of Native Lemon Sole (price: 2.45)</li><li>Whole Maine Lobster, Drawn Butter (price: 5.5)</li><li>Whole Cold Lobster Platter, Garni (price: 4.95)</li><li>Large Cucumber Smelts Fried (price: 1.95)</li><li>Large Cucumber Smelts Broiled (price: 2.25)</li><li>Half Cold Maine Lobster And Broiled Half Chicken (price: 5.5)</li><li>Broiled Boston Scrod (price: 1.95)</li><li>Fish And Chips, English Style (price: 1.95)</li><li>Half Chicken Broiled (price: 4.75)</li><li>Bluefish Saute In Pure Butter (price: 2.25)</li><li>Half Chicken Southern Fried (price: 4.75)</li><li>Broiled Eastern Halibut Steak, Maitre D'hotel (price: 2.45)</li><li>Broiled Florida Red Snapper (price: 2.65)</li><li>Two Brook Trout, Broiled (price: 2.45)</li><li>Mixed Green Salad Bowl</li><li>Rice</li><li>Potatoes</li><li>Two Brook Trout, Saute (price: 2.45)</li><li>Vegetable Du Jour</li><li>Baked Idaho Potato (15c Extra)</li><li>Spanish Or King Mackerel (price: 2.25)</li><li>Hollandaise Sauce (price: 0.6)</li><li>Lobster Sauce (price: 0.75)</li><li>Shrimp Sauce (price: 0.6)</li><li>Fruit Pies</li><li>Egg Sauce (price: 0.4)</li><li>Fresh Fruit Cup</li><li>Creole Sauce (price: 0.25)</li><li>Tartar Sauce (price: 0.15)</li><li>Nesselrode Pie</li><li>Ice Cream</li><li>Roquefort Dressing (price: 0.35)</li><li>Rice Pudding</li><li>Jell O</li><li>Jell O</li><li>Layer Cake</li><li>Half Grapefruit</li><li>Chocolate Pudding</li><li>Sanka</li><li>Postum</li><li>Milk</li><li>Coca Cola</li><li>Coffee Or Tea (Hot Or Cold)</li><li>Broiled Fresh Florida Pompano (price: 3.25)</li><li>Frog's Legs Saute Provencale (price: 2.75)</li><li>Shad Roe</li><li>Boneless Hudson River Shad</li><li>Cold Sea Food Platter Half Lobster, Shrimps, Crab Meat, Oyster And Clam, Garni (price: 3.95)</li><li>Sirloin Steak (price: 5.25)</li><li>Double Rib Lamb Chops (price: 3.5)</li><li>Calf's Liver, Smothered Onions (price: 2.45)</li><li>Filet Mignon With Mushrooms (price: 4.95)</li><li>Pork Chops (price: 2.5)</li><li>Roast Maryland Turkey (price: 2.45)</li><li>Broiled Half Chicken (price: 2.25)</li><li>Fried Half Chicken (price: 2.25)</li><li>Imported Ham Steak, Hawaiian (price: 2.45)</li><li>Smoked Nova Scotia Salmon Sandwich, Bermuda Onion (price: 1.45)</li><li>Chopped Sirloin Of Prime Beef (price: 2.25)</li><li>Shrimp Salad Sandwich (price: 1.25)</li><li>Crab Meat Salad Sandwich (price: 1.25)</li><li>Salmon Salad Sandwich (price: 0.95)</li><li>Fresh Tunafish Sandwich (price: 0.75)</li><li>Bleu Cheese, Served With Crackers (price: 0.35)</li><li>American Cheese Served With Crackers (price: 0.25)</li><li>Swiss Cheese, Served With Crackers (price: 0.35)</li><li>French Ice Cream (Large Scoop) (price: 0.3)</li><li>Iced Coffee Or Tea (price: 0.25)</li><li>Fruit Compote (price: 0.4)</li><li>Coffee Or Tea (price: 0.15)</li><li>Old Fashioned Strawberry Shortcake (price: 0.6)</li><li>Individual Serving Of Milk (price: 0.2)</li><li>Home Made Apple Pie With Cheese (price: 0.35)</li><li>Instant Postum Or Sanka (price: 0.2)</li><li>Fruit Jello With Cream (price: 0.25)</li><li>Rice Pudding With Cream (price: 0.45)</li><li>Fresh Fruit In Season</li><li>Anchor Parfait (price: 0.6)</li><li>Home Made Fresh Fruit Cup (price: 0.45)</li><li>The Seafood Sensation   Broiled Jumbo Florida Shrimps A La Maison   Served With Choice Of Two   French Fried Potatoes, Vegetable Du Jour, Tossed Salad Bowl (price: 2.95)</li><li>Stuffed Whole Boneless Flounder With Lump Maryland Crab Meat Sauce Fantasie   Served With Choice Of Two   Chef's Salad, Vegetable Du Jour, Cole Slaw, French Fried Potatoes (price: 2.95)</li><li>Shrimp Cocktail</li><li>Cherrystone Clams</li><li>Fresh Fruit Cup</li><li>Tomato Juice</li><li>Grapefruit Juice</li><li>Half Grapefruit</li><li>Fish Chowder</li><li>Clam Broth</li><li>Clam Chowder</li><li>Our Famous Original Specialty Deviled Crabettes</li><li>Broiled South African Lobster Tail, Drawn Butter   Lemon Wedge</li><li>Selected Large Half Chicken, Broiled</li><li>Selected Large Half Chicken Southern Fried</li><li>Cole Slaw</li><li>Chef's Salad (Dressing)</li><li>French Fried Potato</li><li>Boiled Potato</li><li>Vegetable Du Jour</li><li>Nesselrode Pie</li><li>Fruit Cup</li><li>Fruit Pies</li><li>Layer Cake</li><li>Home Made Rice Pudding</li><li>Ice Cream</li><li>Home Made Apple Pie</li><li>Fruit Jello</li><li>Oregon Prunes</li><li>Coffee Strip</li><li>Coffee</li><li>Tea</li><li>Seven Up</li><li>Milk</li><li>Coke</li><li>Imported Still Wines Consult Server For Imported Wines</li><li>Frappes, Extra (price: 0.1)</li><li>Orange Blossom Cocktail (price: 0.75)</li><li>Cherry Brandy (price: 0.55- 2.5)</li><li>P. & T. Reserve (price: 0.65)</li><li>Martini (price: 0.75)</li><li>Carstairs (price: 0.65)</li><li>Apricot Brandy (price: 0.55- 20.0)</li><li>Extra Dry Martini (price: 0.85)</li><li>Imperial (price: 0.65)</li><li>Blackberry Brandy (price: 0.55- 3.0)</li><li>Vodka Martini (price: 0.75)</li><li>Peach Brandy (price: 0.55- 15.0)</li><li>Apple Jack (price: 0.6)</li><li>Manhattan (price: 0.75)</li><li>Marschion Bols (price: 0.55- 0.65)</li><li>Jack Rose Cocktail (price: 0.7)</li><li>P.M. (price: 0.65)</li><li>Curacao (price: 0.55- 30.0)</li><li>Daiquiri (price: 0.7)</li><li>Fleischmann's (price: 0.65)</li><li>Anisette (price: 0.55- 85.0)</li><li>Old Fashioned Cocktail (price: 0.7)</li><li>Hunter (price: 0.65)</li><li>Cherry (price: 0.55- 2.0)</li><li>Bacardi Cocktail (price: 0.75)</li><li>Creme De Cacao (price: 0.55- 85.0)</li><li>Gibson (price: 0.85)</li><li>Schenley Reserve (price: 0.7)</li><li>Creme De Menthe (price: 0.55- 15.0)</li><li>Pink Lady (price: 0.8)</li><li>Calvart Reserve (price: 0.7)</li><li>Rock 'N Rye (price: 0.55- 0.65)</li><li>Alexander With Brandy (price: 0.95)</li><li>Seagram's 7 Crown (price: 0.7)</li><li>P. & T, Private Stock (price: 0.75)</li><li>Bloody Mary (price: 0.9)</li><li>Sauterne (Glass) (price: 0.4)</li><li>Chablis (Glass) (price: 0.4)</li><li>Grasshopper (price: 0.9)</li><li>Four Roses (price: 0.75)</li><li>Rhine Wine (Glass) (price: 0.4)</li><li>Screwdriver (price: 0.9)</li><li>Rhine Wine   Half Bottle, Bottle (price: 1.5- 2.75)</li><li>Sauterne   Half Bottle, Bottle (price: 1.5- 2.75)</li><li>Chablis   Half Bottle, Bottle (price: 1.5- 2.75)</li><li>Lord Calvert (price: 0.75)</li><li>Kummel (price: 0.6- 85.0)</li><li>Side Car (price: 0.95)</li><li>Old Overholt (price: 0.85)</li><li>Cointreau (price: 0.65- 85.0)</li><li>Stinger (price: 0.95)</li><li>Southern Comfort (price: 0.75- 85.0)</li><li>Rob Roy (price: 0.9)</li><li>Louis Martini Napa White Wine, Half Bottle, Bottle (price: 1.5- 0.0)</li><li>Champagne Cocktail (price: 1.0)</li><li>Grand Marnier (price: 0.85- 85.0)</li><li>Benedictine (price: 0.9- 20.0)</li><li>B. & B. (price: 0.9- 1.65)</li><li>Cherry Heering (price: 0.9- 85.0)</li><li>Canadian Club (price: 0.8)</li><li>Drambuie (price: 0.95- 85.0)</li><li>Seagram's V.O. (price: 0.8)</li><li>Claret Lemonade (price: 0.65)</li><li>Vodka (price: 0.75)</li><li>Rum Collins (price: 0.75)</li><li>Mc Naughton's (price: 0.75)</li><li>Port (Glass) (price: 0.5)</li><li>Burgundy (Glass) (price: 0.5)</li><li>Claret (Glass) (price: 0.5)</li><li>Tom Collins (price: 0.75)</li><li>Sherry (Glass) (price: 0.5)</li><li>Port   Half Bottle, Bottle (price: 1.5- 0.0)</li><li>Cuba Libre (price: 0.75)</li><li>Burgundy   Half Bottle, Bottle (price: 1.5- 2.75)</li><li>Sherry   Half Bottle, Bottle (price: 1.5- 2.75)</li><li>Claret   Half Bottle, Bottle (price: 1.5- 2.75)</li><li>Gin 'N Tonic (price: 0.75)</li><li>Whiskey Sour (price: 0.75)</li><li>Imported French Brandy (price: 0.9- 0.0)</li><li>Scotch Sour (price: 0.9)</li><li>Metaxa (Greek) Brandy (price: 0.95- 1.1)</li><li>Vodka Collins (price: 0.8)</li><li>Old Forester (price: 0.8)</li><li>Old Taylor (price: 0.8)</li><li>Old Grand Dad (price: 0.8)</li><li>Champagnes Split, Half Bottle, Bottle (price: 1.5- 5.95)</li><li>Native Beer And Ale (price: 0.5)</li><li>Milk Punch (price: 0.75)</li><li>I.W. Harper (price: 0.8)</li><li>Planter's Punch (price: 0.85)</li><li>Imported Beer And Ale (price: 0.7)</li><li>Port Flip (price: 0.75)</li><li>Native Sparkling Burgundy Split, Half Bottle, Bottle (price: 1.5- 0.0)</li><li>Guinness Stout (price: 0.4)</li><li>Vat 69 Scotch (price: 0.75)</li><li>Gin Fizz (price: 0.75)</li><li>J. & B. Scotch (price: 0.8)</li><li>Sloe Gin Fizz (price: 0.75)</li><li>Teacher's Scotch (price: 0.8)</li><li>Nip Of Beer (price: 0.3)</li><li>Sherry Flip (price: 0.65)</li><li>White Horse Scotch (price: 0.8)</li><li>Brandy Flip (price: 1.0)</li><li>Black And White Scotch (price: 0.8)</li><li>Johnnie Walker, Red, Scotch (price: 0.8)</li><li>Ballantine's Scotch (price: 0.8)</li><li>Bacardi P.R. Rum (price: 0.6)</li><li>Dewar's White Label Scotch (price: 0.8)</li><li>Coca Cola (price: 0.2)</li><li>Gilbey's Gin (price: 0.6)</li><li>Johnnie Walker, Black, Scotch (price: 0.9)</li><li>Gordon Gin (price: 0.6)</li><li>Bacardi, Cuba Rum (price: 0.75)</li><li>Canada Dry Ginger Ale Soda (Split) (price: 0.2)</li><li>8 Year Old Scotch (price: 0.9)</li><li>Myer's Jamaica Rum (price: 0.75)</li><li>12 Year Old Scotch (price: 0.95)</li><li>Piper Heidsieck Champagne Per Bottle (price: 11.0)</li><li>Seven Up (price: 0.2)</li><li>Chivas Regal (price: 0.95)</li><li>Mumm's Champagne Per Bottle (price: 11.0)</li><li>Beefeater Gin (price: 0.7)</li><li>Schweppes Tonic (price: 0.2)</li><li>Booth's Imp. Gin (price: 0.7)</li><li>Haig & Haig, Pinch, Scotch (price: 0.9)</li></ul>"
					}
				],
				"seeAlso": []
			}
		]
	},
	{
		"type": "web",
		"url": "http://menus.nypl.org/menus/31054",
		"items": [
			{
				"itemType": "document",
				"title": "Plaza Hotel menu",
				"creators": [],
				"date": "1933",
				"abstractNote": "62 menus bound into 1 volume",
				"callNumber": "1933-0128_wotm",
				"extra": "genre: Restaurant menu\nRestaurant Location: Plaza Hotel\nPhysical Description: 30x21",
				"libraryCatalog": "New York Public Library Menu Collection",
				"url": "http://menus.nypl.org/menu_pages/62720",
				"attachments": [],
				"tags": [],
				"notes": [
					{
						"note": "<h1>Dishes</h1><ul><li>Cherry Stone Clams (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Little Neck Clams (price: 0.45)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Grapefruit (price: 0.65)</li><li>Orange (price: 0.65)</li><li>Melon (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monegasque (price: 0.75)</li><li>Stuffed Celery (price: 0.5)</li><li>Hors D'oeuvre Varies</li><li>Anchovy Canape (price: 0.6)</li><li>Salami (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Hors D'oeuvre Variés</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Wesphalia Ham (price: 1.25)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Home Made Chicken Soup (price: 0.5)</li><li>Bortsch A La Russe (price: 0.5)</li><li>Jellied Gumbo Consomme, Cup (price: 0.4)</li><li>Jellied Consomme, Madrilene, Cup (price: 0.4)</li><li>Jellied Chicken Consomme, Cup (price: 0.4)</li><li>Jellied Beef Consomme, Cup (price: 0.4)</li><li>Planked Boneless Shad And Roe Lady Duveene (price: 1.25)</li><li>Filet Of Sole Cafe De Paris (price: 0.9)</li><li>Brook Trout Au Bleu, Sauce Escoffier (price: 1.0)</li><li>Frogs' Legs A La Cavour (price: 1.25)</li><li>Baked Lobster Beaugency (1/2) (price: 1.5)</li><li>Supreme Of Kingfish Fin De Siecle (price: 1.1)</li><li>Grilled Bluefish Baltimore (price: 1.1)</li><li>Soft Shell Crabs Under Glass Archiduc (price: 1.25)</li><li>Sea Bass Saute Nicoise (price: 1.1)</li><li>Chicken Halibut Au Beurre D'anchois With Potatoes Allumette (price: 1.1)</li><li>Salmon Steak Meuniere With Bananas And Sweet Peppers (price: 1.25)</li><li>Smothered Squab Chicken A La Dickens With Asparagus Tips (1/2) (price: 1.5)</li><li>Roast Hindquarter Of Baby Lamb, Potatoes Parisienne, Succotash Virginia (price: 1.5)</li><li>Glazed Virginia Ham, Sauce Porto With Sweet Potatoes Dixie, Braised Celery (price: 1.25)</li><li>Grilled Kernel Of Sweatbread With Bacon And Heart Of Artichoke Princesse (price: 1.5)</li><li>Escalopines Of Milk Veal Au Vin Blanc With String Beans And Veloute Crecy (price: 1.25)</li><li>Larded Tenderloin Of Beef, Melba (price: 2.0)</li><li>Imperial Squab Souvaroff (price: 2.0)</li><li>Planked Porter House Steak Aux Primeurs (price: 3.0)</li><li>Quenelle Of Capon National (price: 1.25)</li><li>Grilled Egg Plant And Tomato Baltimore (price: 0.9)</li><li>Noisette De Pre Sale Parisienne (price: 1.5)</li><li>Hot House Chicken L'athuile (1/2) (price: 1.75)</li><li>Supreme Of Guinea Hen Tour D'argent (price: 1.75)</li><li>Patty Of Mushrooms A La Reine (price: 1.0)</li><li>Calf's Head En Tortue (price: 1.1)</li><li>Roast Beef (price: 1.0)</li><li>Roast Baby Lamb (price: 1.5)</li><li>Broiler (price: 3.0)</li><li>Broiler (1/2) (price: 1.5)</li><li>Roast Royal Squab (price: 1.75)</li><li>Roast Squab Chicken (price: 2.0)</li><li>Jersey Asparagus (price: 0.9)</li><li>Cauliflower (price: 0.7)</li><li>Beets In Butter (price: 0.5)</li><li>New Peas (price: 0.6)</li><li>New Lima Beans (price: 0.6)</li><li>New String Beans (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>Succotash Plaza (price: 0.65)</li><li>Braised Lettuce (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>White Squash (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Plaza Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Dixie Potatoes (price: 0.5)</li><li>Sunrise Salad (price: 0.6)</li><li>Mimosa Salad (price: 0.6)</li><li>Lettuce, Tomato And Asparagus Tips Salad (price: 0.6)</li><li>Jeannette Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Chicory Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Hot Albufera Pudding (price: 0.35)</li><li>Cold Apple Custard Pudding (price: 0.35)</li><li>Bavarian Cream, Marie Louise (price: 0.4)</li><li>Chocolate Souffle (price: 0.75)</li><li>Orange Jelly (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Fresh Caramel Ice Cream (price: 0.45)</li><li>Orange Sherbet (price: 0.4)</li><li>Strawberry Coupe (price: 0.6)</li><li>Pear Cardinal (price: 0.65)</li><li>Camembert Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Brie Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Stilton Cheese (price: 0.6)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Grapefruit (price: 0.4)</li><li>Orange (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>King Orange (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Fruit Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Melon Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Tomato Monegasque (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Anchovy Canape (price: 0.6)</li><li>Hors D'oeuvre Variés</li><li>Stuffed Celery (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Salami (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Tomato Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Florida Soup With Paillettes Au Chester (price: 0.6)</li><li>Minestrone A L'italienne (price: 0.5)</li><li>Jellied Chicken Consomme, Cup (price: 0.4)</li><li>Jellied Gumbo Consomme, Cup (price: 0.4)</li><li>Jellied Beef Consomme, Cup (price: 0.4)</li><li>Jellied Consomme, Madrilene, Cup (price: 0.4)</li><li>Poached Eggs, Armenonville (price: 0.55)</li><li>Scrambled Eggs, Grand' Mere (price: 0.8)</li><li>Omelette, Agnes Sorel (price: 0.8)</li><li>Fried Soft Shell Crabs With Bacon, Sauce Remoulade (price: 1.25)</li><li>Creamed Salmon In Bordure Florentine (price: 1.25)</li><li>Chicken Halibut, Los Angeles (price: 1.1)</li><li>Kingfish, Amelie (price: 1.1)</li><li>Filet Of Sea Bass Lady Egmont (price: 1.1)</li><li>Brook Trout A La Lindbergh (price: 1.1)</li><li>Grilled Bluefish, Jardiniere (price: 1.1)</li><li>Boneless Shad And Roe Beaucaire (price: 1.25)</li><li>Supreme Of Sole Commodore (price: 0.9)</li><li>Frogs' Legs Rissolees Royer (price: 1.25)</li><li>Boiled Short Rib Of Beef, New England Style (price: 1.1)</li><li>Braised Rack Of Milk Veal Bourgeoise With Peas And Fresh Noodles (price: 1.5)</li><li>Roast Loin Of Baby Lamb With Lima Beans Bretonne (price: 1.5)</li><li>Grilled Squab Chicken With Bacon, Mushrooms, Marrow, Potatoes Souffles (1/2) (price: 1.5)</li><li>Fresh Vegetable Luncheon Au Gratin With Poached Egg (price: 1.25)</li><li>Fresh Mushrooms In Crust Maryland (price: 1.0)</li><li>Imperial Squab A L'americaine (price: 2.0)</li><li>Supreme Of Guinea Hen Rossini (price: 1.75)</li><li>Chicken Saute Lafayette (1/2) (price: 1.75)</li><li>Egg With Ham (price: 0.5)</li><li>Egg, Nicoise Or A La Russe (price: 0.75)</li><li>Cold Egg Finlandaise (price: 0.6)</li><li>Cold Halibut Steak A L'espagnole (price: 1.25)</li><li>Cold Kennebec Salmon Parisienne (price: 1.25)</li><li>Crab Ravigote (price: 1.0)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Cold Shrimps A La Joseph (price: 0.75)</li><li>Cold Chicken And Ham Pie (price: 1.5)</li><li>Cold Supreme Jeannette (price: 1.25)</li><li>Cold Supreme Plaza (price: 1.75)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Jersey Asparagus (price: 0.9)</li><li>Cauliflower (price: 0.7)</li><li>Beets In Butter (price: 0.5)</li><li>String Beans (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>Carrots Vichy (price: 0.5)</li><li>Braised Celery (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Hubbard Squash (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Mashed Yellow Turnips (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Plaza Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Lettuce, Eggs And Beets Salad (price: 0.6)</li><li>Orleans Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Endive Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Duchesse Pudding (price: 0.35)</li><li>Deep Dish Rhubarb Pie (price: 0.4)</li><li>Hazelnut Custard Pie (price: 0.35)</li><li>Chocolate Layer Cake (price: 0.4)</li><li>Raspberry Slice (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Strawberry Ice (price: 0.4)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Lemon Sherbet (price: 0.4)</li><li>Vanilla Mousse, Hot Chocolate Sauce (price: 0.6)</li><li>Parfait Tosca (price: 0.6)</li><li>Brie Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Stilton Cheese (price: 0.6)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Orange (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Pear (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Filet Of Herring (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Smoked Salmon (price: 0.75)</li><li>Hors D'oeuvre Variés</li><li>Stuffed Celery (price: 0.5)</li><li>Anchovy Canape (price: 0.6)</li><li>Antipasto (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Salami (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Soup, Chicken Printanier With Barley (price: 0.5)</li><li>Cream Of New Peas Boula Boula Soup (price: 0.6)</li><li>Jellied Consomme, Chicken (price: 0.4)</li><li>Jellied Consomme, Beef (price: 0.4)</li><li>Gumbo</li><li>Jellied Consomme, Madrilene (price: 0.4)</li><li>Supreme Of Sole Duchesse (price: 0.9)</li><li>Poached Kennebec Salmon, Sauce Joinville (price: 1.1)</li><li>Soft Shell Crabs Rissolées With Almonds (price: 1.25)</li><li>Chicken Halibut à La Turque (price: 1.1)</li><li>Boneless Shad And Roe Belle Meunière (price: 1.25)</li><li>Frilled Bluefish Pompadour (price: 1.1)</li><li>Baked Lobster In Shell Majestic (1/2) (price: 1.5)</li><li>Sea Bass Sauté Alphonse Xiii (price: 1.1)</li><li>Patty Of Frogs' Legs Poulette (price: 1.25)</li><li>Brook Trout A La Canajoharie (price: 1.1)</li><li>Roast Rack And Loin Of Baby Lamb Nivernaise With Potatoes Parisienne (price: 1.75)</li><li>Supreme Of Hot House Chicken, Rimini, Heart Of Artichoke Princesse (price: 1.75)</li><li>Escalopine Of Milk Veal à L'estragon, Noodles In Butter, Mashed Carrot (price: 1.25)</li><li>Larded Kernel Of Sweetbread Prince De Galles (price: 1.5)</li><li>Petit Tournedos Sauté Chantecler (price: 1.75)</li><li>Grilled Fresh Mushrooms On Toast With Ham Au Beurre D'escargots (price: 1.0)</li><li>Poussin De Serre En Vol Au Vent Financiere (price: 2.25)</li><li>Imperial Squab En Compote (price: 2.0)</li><li>Quenelles Of Ham Florentine (price: 1.1)</li><li>Noisette Of Spring Lamb Judic (price: 1.5)</li><li>Calf's Head En Tortue (price: 1.1)</li><li>Roast Baby Lamb (price: 1.5)</li><li>Roast Beef (price: 1.0)</li><li>Broiler (price: 1.5- 3.0)</li><li>Roast Royal Squab (price: 1.75)</li><li>Roast Squab Chicken (price: 2.0)</li><li>Egg With Ham (price: 0.5)</li><li>Egg, Nicoise (price: 0.75)</li><li>Egg Finlandaise (price: 0.6)</li><li>Egg A La Russe (price: 0.75)</li><li>Kennebec Salmon Parisienne (price: 1.25)</li><li>Halibut Steak A L'espagnole (price: 1.25)</li><li>Crab Ravigote (price: 1.0)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Shrimps A La Joseph (price: 0.75)</li><li>Supreme Jeannette (price: 1.25)</li><li>Chicken And Ham Pie (price: 1.5)</li><li>Supreme Plaza (price: 1.75)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Jersey Asparagus (price: 0.9)</li><li>Cauliflower (price: 0.7)</li><li>Beets In Butter (price: 0.5)</li><li>New Peas (price: 0.6)</li><li>String Beans (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>Succotash Plaza (price: 0.65)</li><li>Braised Lettuce (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Stewed Okra (price: 0.65)</li><li>Artichoke (price: 0.6)</li><li>White Squash (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Fried Egg Plant (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Voisin Potatoes (price: 0.5)</li><li>Hartford Salad (price: 0.6)</li><li>Juliette Salad (price: 0.6)</li><li>Excelsior Salad (price: 0.6)</li><li>Royal Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Endive Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Hot Tapioca Pudding (price: 0.35)</li><li>Cold Pear Custard Pudding (price: 0.35)</li><li>Brandied Peach Jubliee (price: 0.9)</li><li>Souffle Rothschild (price: 0.75)</li><li>Crepes Georgette (price: 1.0)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Orange Sherbet (price: 0.4)</li><li>Strawberry Coupe (price: 0.6)</li><li>Pear Cardinal (price: 0.65)</li><li>Brie Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>King Orange (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Pear (price: 0.2)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Melon Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Filet Of Herring (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monegasque (price: 0.75)</li><li>Hors D'oeuvre Variés</li><li>Stuffed Celery (price: 0.5)</li><li>Anchovy Canape (price: 0.6)</li><li>Westphalia Ham (price: 1.25)</li><li>Antipasto (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Westmoreland Soup (price: 0.4)</li><li>Leek And Potato Soup (price: 0.5)</li><li>Jellied Consomme, Beef (price: 0.4)</li><li>Jellied Consomme, Madrilene (price: 0.4)</li><li>Jellied Consomme, Chicken (price: 0.4)</li><li>Gumbo (price: 0.4)</li><li>Poached Eggs With Succotash Au Gratin (price: 0.55)</li><li>Omelette, Savoyarde (price: 0.8)</li><li>Scrambled Eggs With Calf's Brains (price: 0.8)</li><li>Fried Codfish Cakes With Bacon, Sauce Orly (price: 0.9)</li><li>Aiguillette Of Kingfish Sarah (price: 1.1)</li><li>Boneless Shad And Roe Florida (price: 1.25)</li><li>Soft Shell Crabs On Toast, Beurre Ravigotte (price: 1.0)</li><li>Brook Trout Genevoise (price: 1.1)</li><li>Supreme Of Sole Georgette (price: 0.9)</li><li>Filet Of Sea Bass Cancale (price: 1.1)</li><li>Lobster Thermidor (1/2) (price: 1.5)</li><li>Frogs' Legs Bordelaise With Rice Pilaw (price: 1.25)</li><li>Browned Roast Beef Hash With Fried Egg And Spaghetti Milanaise (price: 0.9)</li><li>Grilled Milk Veal Chop Colerette (price: 1.25)</li><li>Baked Shell Of Capon A La Royal (price: 1.5)</li><li>Honeycomb Tripe A La Creole With Parsley Potatoes (price: 1.1)</li><li>Roast Hindquarter Of Baby Lamb Cullinan (price: 1.5)</li><li>Sugar Cured Ham Steak, Southern Style (price: 1.25)</li><li>Assorted Fresh Vegetable Luncheon (price: 1.25)</li><li>Patty Of Mushrooms A La Newburg (price: 1.0)</li><li>Hot House Chicken A La Marengo (1/2) (price: 1.75)</li><li>Imperial Squab A La Bressane (price: 2.0)</li><li>Supreme Of Guinea Hen Au Foie Gras (price: 1.75)</li><li>Cold Egg Finlandaise (price: 0.6)</li><li>Egg, Nicoise (price: 0.75)</li><li>Egg A La Russe (price: 0.75)</li><li>Cold Egg With Ham (price: 0.5)</li><li>Cold Halibut Steak A L'espagnole (price: 1.25)</li><li>Cold Kennebec Salmon Parisienne (price: 1.25)</li><li>Cold Shrimps A La Joseph (price: 0.75)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Crab Ravigote (price: 1.0)</li><li>Cold Chicken And Ham Pie (price: 1.5)</li><li>Cold Supreme Jeannette (price: 1.25)</li><li>Cold Supreme Plaza (price: 1.75)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Assorted Cold Cuts Wit Chicken (price: 1.75)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Beets In Butter (price: 0.5)</li><li>Jersey Asparagus (price: 0.9)</li><li>Cauliflower (price: 0.7)</li><li>Carrots Vichy (price: 0.5)</li><li>Succotash Plaza (price: 0.65)</li><li>String Beans (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Creamed Spinach (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Sweet Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Blackstone Salad (price: 0.6)</li><li>Lettuce, Orange And Grapefruit Salad (price: 0.6)</li><li>Knickerbocker Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Rocheford Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Lettuce Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Apricot Pie (price: 0.35)</li><li>Vanilla Custard Pie (price: 0.35)</li><li>White Cherry Puddiing (price: 0.35)</li><li>Apple Tart (price: 0.4)</li><li>Punch Cake (price: 0.4)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Rice Pudding (price: 0.35)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Parfait Tosca (price: 0.6)</li><li>Lemon Sherbet (price: 0.4)</li><li>Vanilla Mousse, Hot Chocolate Sauce (price: 0.6)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Camembert Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Strawberries (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>King Orange (price: 0.2)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Orange Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Hors D'oeuvre Variés</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Beef Broth A L'ecossaise (price: 0.5)</li><li>Cream Of Mushrooms Soup Aux Souffles (price: 0.5)</li><li>Jellied Beef Consomme, Cup (price: 0.4)</li><li>Jellied Gumbo Consomme, Cup (price: 0.4)</li><li>Jellied Consomme, Madrilene, Cup (price: 0.4)</li><li>Jellied Chicken Consomme, Cup (price: 0.4)</li><li>Soft Shell Crabs A La Creole (price: 1.25)</li><li>Boiled Chicken Halibut, Sauce Mousseline (price: 1.1)</li><li>Supreme Of Sole Mervina (price: 0.9)</li><li>Baked Lobster Grand Hotel (1/2) (price: 1.5)</li><li>Turban Of Frogs' Legs Regence (price: 1.5)</li><li>Brook Trout Matelotte (price: 1.1)</li><li>Grilled Bluefish With Bacon Au Capeco (price: 1.1)</li><li>Boneless Shad And Roe A L'orange (price: 1.25)</li><li>Carne Of Salmon Montgolfier (price: 1.25)</li><li>Sliced Sirloin Of Beef Saute Mexicaine (price: 1.75)</li><li>Smothered Squab Chicken Archiduc, Veloute Crecy, Asparagus Tips (1/2) (price: 1.5)</li><li>Grilled Kernel Of Sweetbread With Ham And Succotash Plaza (price: 1.5)</li><li>Braised Milk Veal Zingara With Ravioli Bercetto (price: 1.25)</li><li>Boneless Saddle Of Baby Lamb Ambassadeur (price: 1.75)</li><li>Grilled Mushrooms On Toast With Bacon, Beurre Ravigotte (price: 1.0)</li><li>Arroz Con Pollo A L'espagnole (1/2) (price: 1.75)</li><li>Mousseline Of Ham National (price: 1.1)</li><li>Boneless Royal Squab A La Turner (price: 2.0)</li><li>Supreme Of Chicken Champs Elysees (price: 1.75)</li><li>Roast Beef (price: 1.0)</li><li>Roast Baby Lamb (price: 1.5)</li><li>Roast Squab Chicken (price: 2.0)</li><li>Broiler (1/2) (price: 1.5)</li><li>Roast Royal Squab (price: 1.75)</li><li>Broiler (price: 3.0)</li><li>Cold Egg Finlandaise (price: 0.6)</li><li>Cold Egg With Ham (price: 0.5)</li><li>Egg, Nicoise (price: 0.75)</li><li>Egg A La Russe (price: 0.75)</li><li>Cold Halibut Steak A L'espagnole (price: 1.25)</li><li>Cold Kennebec Salmon Parisienne (price: 1.25)</li><li>Cold Shrimps A La Joseph (price: 0.75)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Crab Ravigote (price: 1.0)</li><li>Cold Chicken And Ham Pie (price: 1.5)</li><li>Cold Supreme Jeannette (price: 1.25)</li><li>Cold Supreme Plaza (price: 1.75)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Jersey Asparagus (price: 0.9)</li><li>Beets In Butter (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>Lima Beans (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>New Peas (price: 0.6)</li><li>String Beans (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>Stewed Okra (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Creamed Spinach (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Sweet Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Dixie Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Kuroki Salad (price: 0.6)</li><li>Louisette Salad (price: 0.6)</li><li>Lettuce, Tomato And Cucumber Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Rocheford Salad (price: 0.6)</li><li>Lettuce Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Cold Pudding Suchard (price: 0.35)</li><li>Hot Montmorency Pudding (price: 0.35)</li><li>Fruits In Jelly (price: 0.35)</li><li>Rice Maltaise (price: 0.4)</li><li>Strawberry Souffle (price: 0.75)</li><li>French Pastry (1) (price: 0.25)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Orange Sherbet (price: 0.4)</li><li>Strawberry Coupe (price: 0.6)</li><li>Pear Cardinal (price: 0.65)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Brie Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Pear (price: 0.2)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Melon Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Filet Of Herring (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monegasque (price: 0.75)</li><li>Stuffed Celery (price: 0.5)</li><li>Hors D'oeuvre Variés</li><li>Anchovy Canape (price: 0.6)</li><li>Salami (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Antipasto (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Cream Of Sorrel Soup With Rice (price: 0.5)</li><li>Crab Meat Gumbo (price: 0.5)</li><li>Jellied Consomme, Chicken (price: 0.4)</li><li>Jellied Consomme, Madrilene (price: 0.4)</li><li>Jellied Consomme, Beef (price: 0.4)</li><li>Gumbo (price: 0.4)</li><li>Shirred Eggs A La Turque (price: 0.5)</li><li>Omelette Parisienne (price: 0.8)</li><li>Poached Eggs Benedict (price: 0.6)</li><li>Baked Boston Scrod A La Dickens (price: 0.9)</li><li>Supreme Of Sole Breval (price: 0.9)</li><li>Chicken Halibut A La Bartelette (price: 1.1)</li><li>Frogs' Legs Rissolees Provencale (price: 1.0)</li><li>Sea Bass Meuniere Alphonse Xiii (price: 1.1)</li><li>Brook Trout A La Richebourg (price: 1.1)</li><li>Fried Soft Shell Crabs Gribiche (price: 1.0)</li><li>Boneless Shad And Roe, Los Angeles (price: 1.25)</li><li>Prime Rib Of Beef With Veloute Freneuse (price: 1.25)</li><li>Stuffed Breast Of Veal With New Succotash, Potatoes Olivette Persillee (price: 1.1)</li><li>Grilled Squab Chicken A La Plaza (1/2) (price: 1.5)</li><li>Pork Chop Saute Milanaise (price: 1.25)</li><li>Roast Rack Of Baby Lamb Boulangere (price: 1.5)</li><li>Fresh Vegetable Luncheon En Bordure, Mornay (price: 1.25)</li><li>Poussin De Serre Sous La Cendre Perigourdine (price: 2.25)</li><li>Grilled Jumbo Squab En Crapaudine (price: 1.75)</li><li>Hot House Chicken L'athuile (1/2) (price: 1.75)</li><li>Steak Minute With Cepes Provencale (price: 1.75)</li><li>Supreme Of Guinea Hean A L'orange (price: 1.75)</li><li>Egg Finlandaise (price: 0.6)</li><li>Egg With Ham (price: 0.5)</li><li>Egg A La Russe (price: 0.75)</li><li>Egg, Nicoise (price: 0.75)</li><li>Kennebec Salmon Parisienne (price: 1.25)</li><li>Halibut Steak A L'espagnole (price: 1.25)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Shrimps A La Joseph (price: 0.75)</li><li>Crab Ravigote (price: 1.0)</li><li>Chicken And Ham Pie (price: 1.5)</li><li>Supreme Jeannette (price: 1.25)</li><li>Supreme Plaza (price: 1.75)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Beets In Butter (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>Jersey Asparagus (price: 0.8)</li><li>String Beans (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>Succotash Plaza (price: 0.65)</li><li>Lima Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>White Squash (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Mashed Yellow Turnips (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Voisin Potatoes (price: 0.5)</li><li>Louise Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Richard Salad (price: 0.6)</li><li>Lettuce, Cucumber And Beets Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Romaine Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Red Sour Cherry Pie (price: 0.35)</li><li>Orange Custard Pie (price: 0.35)</li><li>Mylady Pudding (price: 0.35)</li><li>Apple Fritters, Apricot Sauce (price: 0.45)</li><li>Hazelnut Layer Cake (price: 0.4)</li><li>French Pastry (1) (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Lemon Sherbet (price: 0.4)</li><li>Parfait Tosca (price: 0.6)</li><li>Vanilla Mousse, Hot Chocolate Sauce (price: 0.6)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Camembert Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Roquefort Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Cocktail Sauce (price: 0.1)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Orange Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Filet Of Herring (price: 0.5)</li><li>Tomato Monegasque (price: 0.75)</li><li>Smoked Salmon (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Anchovy Canape (price: 0.6)</li><li>Hors D'oeuvre Variés</li><li>Stuffed Celery (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Cream Of Vegetables Soup Encore (price: 0.5)</li><li>Mutton Broth Scotch Style (price: 0.5)</li><li>Jellied Chicken Consomme, Cup (price: 0.4)</li><li>Jellied Beef Consomme, Cup (price: 0.4)</li><li>Jellied Madrilene Consomme, Cup (price: 0.4)</li><li>Jellied Gumbo Consomme, Cup (price: 0.4)</li><li>Boiled Live Codfish, Mousseline Sauce (price: 0.9)</li><li>Filet Of Kingfish, Rosine (price: 1.1)</li><li>Paupiette Of Sole A La Royal (price: 0.9)</li><li>Frogs' Legs Bercy With Rice Pilaw (price: 1.25)</li><li>Baked Lobster Vanderbilt (1/2) (price: 1.5)</li><li>Brook Trout Maconnaise (price: 1.1)</li><li>Sea Bass Saute Doria (price: 1.1)</li><li>Shad Roe A La Tailer (price: 1.5)</li><li>Soft Shell Crabs A La Newburg With Mushrooms (price: 1.25)</li><li>Roast Baron Of Baby Lamb With Potatoes And Vegetables Armenonville (price: 1.5)</li><li>Squab Chicken Poele Metternick (1/2) (price: 1.5)</li><li>Escalopine Of Milk Veal Mascotte (price: 1.25)</li><li>Grilled Kernel Of Sweetbread On Toast With Bacon St. Germain (price: 1.5)</li><li>Braised Beef A La Mode With Green Noodles (price: 1.1)</li><li>Noisette De Pre Sale Rossini (price: 1.5)</li><li>Mousseline Of Ham Carignan (price: 1.1)</li><li>Calf's Head Vinaigrette (price: 0.9)</li><li>Poussin De Serre A La Souvaroff (price: 2.25)</li><li>Breast Of Guinea Hen Santa Clara (price: 1.75)</li><li>Boneless Imperial Squab A La Turner (price: 2.0)</li><li>Stuffed Mushrooms Under Glass Bercy (price: 1.0)</li><li>Roast Beef (price: 1.0)</li><li>Roast Baby Lamb (price: 1.5)</li><li>Broiler Chicken (1/2) (price: 1.5)</li><li>Broiler Chicken (price: 3.0)</li><li>Roast Squab Chicken (price: 2.0)</li><li>Roast Royal Squab (price: 1.75)</li><li>Egg With Ham (price: 0.5)</li><li>Egg Finlandaise (price: 0.6)</li><li>Egg, Nicoise (price: 0.75)</li><li>Egg, A La Russe (price: 0.75)</li><li>Kennebec Salmon Parisienne (price: 1.25)</li><li>Halibut Steak A L'espagnole (price: 1.25)</li><li>Shrimps A La Joseph (price: 0.75)</li><li>Crab Ravigote (price: 1.0)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Supreme Jeannette (price: 1.25)</li><li>Chicken And Ham Pie (price: 1.5)</li><li>Supreme Plaza (price: 1.75)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Beets In Butter (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>Jersey Asparagus (price: 0.8)</li><li>Carrots Vichy (price: 0.5)</li><li>Succotash Plaza (price: 0.65)</li><li>Lima Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>String Beans (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Voisin Potatoes (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Dixie Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Sweet Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Lettuce, Celery And Pimentos Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Bonnie Brea Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Tomato Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Cold Pudding Sans Gene (price: 0.35)</li><li>Hot Fig Pudding (price: 0.35)</li><li>Souffle Surprise (price: 0.75)</li><li>Raspberry Jelly (price: 0.35)</li><li>Almond Blanc Manger (price: 0.4)</li><li>French Pastry (1) (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>Strawberry Ice (price: 0.4)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Strawberry Coupe (price: 0.6)</li><li>Pear Cardinal (price: 0.65)</li><li>Orange Sherbet (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Camembert Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Roquefort Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Filet Of Herring (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monegasque (price: 0.75)</li><li>Anchovy Canape (price: 0.6)</li><li>Hors D'oeuvre Variés</li><li>Stuffed Celery (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Antipasto (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Chicken Okra Soup (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Tomato Soup (price: 0.4)</li><li>Cream Soup Mille Fanti (price: 0.5)</li><li>Onion Soup Gratine (price: 0.5)</li><li>Jellied Consomme, Chicken (price: 0.4)</li><li>Jellied Consomme, Madrilene (price: 0.4)</li><li>Jellied Consomme, Beef (price: 0.4)</li><li>Gumbo (price: 0.4)</li><li>Poached Eggs A La Henry (price: 0.75)</li><li>Omelette Au Gruyere (price: 0.8)</li><li>Scramble Eggs With Asparagus Tips (price: 0.8)</li><li>Steamed Finnan Haddie Des Gourmets (price: 0.9)</li><li>Filet Of Flounder St. Germain (price: 0.9)</li><li>Frogs' Legs Rissolees A La Royer (price: 1.25)</li><li>Soft Shell Crabs A La Sterry (price: 1.25)</li><li>Chicken Halibut A La Manhatta (price: 1.1)</li><li>Sea Bass Saute A L'orange (price: 1.1)</li><li>Grilled Boneless Shad With Mushrooms, Bacon And Cole Slaw (price: 1.25)</li><li>Brook Trout Cleopatra (price: 1.1)</li><li>Plate Corned Beef With Spinach, Parsley Potatoes And Chow Chow (price: 1.1)</li><li>Roast Hindquarter Of Baby Lamb With White Beans Bretonne (price: 1.5)</li><li>Milk Veal Cutlet Reforme With Carrots And Peas (price: 1.25)</li><li>Assorted Fresh Vegetables En Bordure Gratinee (price: 1.25)</li><li>Shell Of Capon A La Royal (price: 1.5)</li><li>English Mutton Chop Vert Pre (price: 1.25)</li><li>Smothered Guinea Hen, Marengo (1/2) (price: 1.75)</li><li>Supreme Of Chicken Belle Helene (price: 1.75)</li><li>Planked Small Steak Aux Primeurs (price: 2.75)</li><li>Grilled Fresh Mushrooms On Toast, Beurre D'escargots (price: 0.8)</li><li>Cold Egg With Ham (price: 0.5)</li><li>Egg Nicoise (price: 0.75)</li><li>Cold Egg A La Russe (price: 0.75)</li><li>Cold Egg Finlandaise (price: 0.6)</li><li>Cold Kennebec Salmon Parisienne (price: 1.25)</li><li>Cold Halibut Steak A L'espagnole (price: 1.25)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Crab Ravigote (price: 1.0)</li><li>Cold Shrimps A La Joseph (price: 0.75)</li><li>Cold Chicken And Ham Pie (price: 1.5)</li><li>Cold Supreme Jeannette (price: 1.25)</li><li>Cold Supreme Plaza (price: 1.75)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Jersey Asparagus (price: 0.8)</li><li>Cauliflower (price: 0.7)</li><li>Beets In Butter (price: 0.5)</li><li>Lima Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>String Beans (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>Carrots Vichy (price: 0.5)</li><li>Braised Lettuce (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>White Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Voisin Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Jeannette Salad (price: 0.6)</li><li>Mimosa Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Lettuce, Tomato And Asparagus Tips Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Mixed Green Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Macaroon Custard Pie (price: 0.35)</li><li>Apple Dumpling, Hard Sauce (price: 0.4)</li><li>Peach Pie (price: 0.35)</li><li>Pompadour Chantilly (price: 0.25)</li><li>Marquise Layer Cake (price: 0.4)</li><li>French Pastry (1) (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>Strawberry Ice (price: 0.4)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Parfait Tosca (price: 0.6)</li><li>Vanilla Mousse, Hot Chocolate Sauce (price: 0.6)</li><li>Lemon Sherbet (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Camembert Cheese (price: 0.45)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Delicious Apple (price: 0.2)</li><li>Orange (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>King Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Little Neck Clams (price: 0.45)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Fruit Supreme (price: 0.65)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Filet Of Herring (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monegasque (price: 0.75)</li><li>Anchovy Canape (price: 0.6)</li><li>Hors D'oeuvre Variés</li><li>Stuffed Celery (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Cream Of Asparagus Soup Aux Quenelles (price: 0.5)</li><li>Florida Soup With Paillettes Doree (price: 0.6)</li><li>Jellied Madrilene Consomme, Cup (price: 0.4)</li><li>Jellied Gumbo Consomme, Cup (price: 0.4)</li><li>Jellied Chicken Consomme, Cup (price: 0.4)</li><li>Jellied Beef Consomme, Cup (price: 0.4)</li><li>Boiled Kennebec Salmon, Sauce Hollandaise (price: 1.1)</li><li>Supreme Of Sole A La Plaza (price: 0.9)</li><li>Patty Of Frogs' Legs With Mushrooms Newburg (price: 1.25)</li><li>Brook Trout A La Tivoli (price: 1.1)</li><li>Grilled Bluefish With Cucumber Vernon (price: 1.1)</li><li>Sea Bass Saute Dauphine (price: 1.1)</li><li>Sof Shell Crabs Rissolees With Almonds And Apples (price: 1.25)</li><li>Boneless Shad And Roe Meuniere A La Vichy (price: 1.25)</li><li>Roast Hindquarter Of Baby Lamb With Succotash Virginia, Potatoes Olivette (price: 1.5)</li><li>Smothered Breast Of Chicken Archiduc With Asparagus Tips, Mashed Carrots (price: 1.75)</li><li>Escalopine Of Milk Veal A La Kossuth With Veloute Of Artichokes (price: 1.25)</li><li>Steak Minute Saute Chasseur With Noodles And Braised Celery (price: 1.75)</li><li>Grilled Kernel Of Sweetbread Gismonde (price: 1.5)</li><li>Mousseline Of Ham Florentine (price: 1.1)</li><li>Milk Fed Chicken Saute Bordelaise (1/2) (price: 1.75)</li><li>Noisette De Pre Sale Rossini (price: 1.5)</li><li>Imperial Squab En Casserole Turner (price: 2.0)</li><li>Grilled Fresh Mushrooms On Toast With Ham Maitre D'hotel (price: 1.0)</li><li>Roast Baby Lamb (price: 1.5)</li><li>Roast Beef (price: 1.0)</li><li>Roast Royal Squab (price: 1.75)</li><li>Broiled Chicken (price: 3.0)</li><li>Broiler Chicken (1/2) (price: 1.5)</li><li>Roast Squab Chicken (price: 2.0)</li><li>Egg, Nicoise (price: 0.75)</li><li>Cold Egg With Ham (price: 0.5)</li><li>Cold Egg A La Russe (price: 0.75)</li><li>Cold Egg Finlandaise (price: 0.6)</li><li>Cold Halibut Steak A L'espagnole (price: 1.25)</li><li>Cold Kennebec Salmon Parisienne (price: 1.25)</li><li>Cold Shrimps A La Joseph (price: 0.75)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Crab Ravigote (price: 1.0)</li><li>Cold Supreme Plaza (price: 1.75)</li><li>Cold Chicken And Ham Pie (price: 1.5)</li><li>Cold Supreme Jeannette (price: 1.25)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Cauliflower (price: 0.7)</li><li>Beets In Butter (price: 0.5)</li><li>Jersey Asparagus (price: 0.8)</li><li>Lima Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>String Beans (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>Stewed Okra (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Creamed Spinach (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Sweet Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Sunrise Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Lily Salad (price: 0.6)</li><li>Plaza Special Salad (price: 0.6)</li><li>Princess Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Rocheford Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Tomato Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Cold Farina Pudding (price: 0.35)</li><li>Hot Valencia Pudding (price: 0.35)</li><li>Vanilla Bavaroise (price: 0.4)</li><li>Apricot Bourdaloue (price: 0.4)</li><li>Souffle Nougatine (price: 0.75)</li><li>French Pastry (1) (price: 0.25)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>Strawberry Ice (price: 0.4)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Pear Cardinal (price: 0.65)</li><li>Orange Sherbet (price: 0.4)</li><li>Strawberry Coupe (price: 0.6)</li><li>Swiss Cheese (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Orange (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Grapefruit (price: 0.4)</li><li>King Orange (price: 0.2)</li><li>Little Neck Clams (price: 0.45)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Pineapple Paradise (price: 0.75)</li><li>Fruit Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monegasque (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Stuffed Celery (price: 0.5)</li><li>Anchovy Canape (price: 0.6)</li><li>Hors D'oeuvre Variés</li><li>Westphalia Ham (price: 1.25)</li><li>Salami (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Jubilee Soup (price: 0.5)</li><li>Clam Chowder Manhattan (price: 0.5)</li><li>Jellied Beef Consomme, Cup (price: 0.4)</li><li>Jellied Madrilene Consomme, Cup (price: 0.4)</li><li>Jellied Chicken Consomme, Cup (price: 0.4)</li><li>Jellied Gumbo Consomme, Cup (price: 0.4)</li><li>Poached Eggs, Lorenzo (price: 0.75)</li><li>Omelet With Chicken Livers Madere (price: 0.8)</li><li>Shirred Eggs Bercy (price: 0.5)</li><li>Boston Rock Cod Saute Lyonnaise (price: 0.9)</li><li>Chicken Halibut A La Venitienne (price: 1.1)</li><li>Filet Of Kingfish, Laperouse (price: 1.1)</li><li>Supreme Of Sole Waleska (price: 0.9)</li><li>Sea Bass Saute Cleopatra (price: 1.1)</li><li>Brook Trout A La Beaucaire (price: 1.1)</li><li>Crab Flakes A La King, Wheat Cakes (price: 1.25)</li><li>Boneless Shad And Roe, Tyrolienne (price: 1.25)</li><li>Frogs' Legs Bercy With Rice Pilaw (price: 1.25)</li><li>Prime Rib Of Beef With Potatoes Fondantes And Stuffed Tomato (price: 1.25)</li><li>Braised Milk Veal Chop With Spinach And Mushrooms (price: 1.25)</li><li>Roast Loin Of Baby Lamb With Cauliflower Polonaise (price: 1.5)</li><li>Smothered Squab Chicken A La Biarritz (1/2) (price: 1.5)</li><li>Stuffed Mushrooms Under Glass Maitre D'hotel (price: 1.0)</li><li>Fresh Vegetable Luncheon With Poached Egg (price: 1.25)</li><li>Imperial Squab Sous La Cendre Perigourdine (price: 2.0)</li><li>Mixed Grill A L'americaine (price: 1.25)</li><li>Supreme Of Guinea Hen D'alcantora (price: 1.75)</li><li>Milk Fed Chicken Lathuile (1/2) (price: 1.75)</li><li>Filet Mignon Henry Iv (price: 2.0)</li><li>Egg Finlandaise (price: 0.6)</li><li>Egg A La Russe (price: 0.75)</li><li>Egg With Ham (price: 0.5)</li><li>Egg, Nicoise (price: 0.75)</li><li>Cold Halibut Steak A L'espagnole (price: 1.25)</li><li>Kennebec Salmon Parisienne (price: 1.25)</li><li>Crab Ravigote (price: 1.0)</li><li>Shrimps A La Joseph (price: 0.75)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Supreme Jeannette (price: 1.25)</li><li>Chicken And Ham Pie (price: 1.5)</li><li>Supreme Plaza (price: 1.75)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Beets In Butter (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>Jersey Asparagus (price: 0.8)</li><li>Carrots Vichy (price: 0.5)</li><li>String Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>Artichoke (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>White Squash (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Voisin Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Sweet Potatoes (price: 0.5)</li><li>Juliette Salad (price: 0.6)</li><li>Excelsior Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Royal Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Rocheford Salad (price: 0.6)</li><li>Escarole Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Strawberry Meringue Pie (price: 0.35)</li><li>Farina Pudding, Fruit Sauce (price: 0.35)</li><li>Boston Cream Puff (price: 0.25)</li><li>Vanilla Custard Pie (price: 0.35)</li><li>Flemish Apple Cake (price: 0.4)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Strawberry Ice (price: 0.4)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Parfait Tosca (price: 0.6)</li><li>Vanilla Mousse, Hot Chocolate Sauce (price: 0.6)</li><li>Lemon Sherbet (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Stilton Cheese (price: 0.6)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Orange (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Grapefruit (price: 0.4)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Little Neck Clams (price: 0.45)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Fruit Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Hors D'oeuvre Variés</li><li>Stuffed Celery (price: 0.5)</li><li>Anchovy Canape (price: 0.6)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Chicken Okra Soup (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Tomato Soup (price: 0.4)</li><li>Soup, Petite Marmite St. Michel (price: 0.6)</li><li>Soup, Cream Of Shrimps Aux Quenelles (price: 0.5)</li><li>Jellied Consomme, Beef (price: 0.4)</li><li>Jellied Consomme, Madrilene (price: 0.4)</li><li>Gumbo (price: 0.4)</li><li>Jellied Consomme, Chicken (price: 0.4)</li><li>Boiled Live Codfish, Egg Sauce (price: 0.9)</li><li>Stuffed Brook Trout Polonaise (price: 1.1)</li><li>Supreme Of Sole Amelie (price: 0.9)</li><li>Soft Shell Crabs, Belle Meuniere (price: 1.25)</li><li>Grilled Bluefish With Cucumber Vernon (price: 1.1)</li><li>Lobster Excelsior (1/2) (price: 1.5)</li><li>Halibut Steak Meuniere A L'orange (price: 1.1)</li><li>Patty Of Frogs' Legs Maryland (price: 1.25)</li><li>Planked Boneless Shad, Bouquetiere (price: 1.25)</li><li>Filet Of Sea Bass Breval (price: 1.1)</li><li>Roast Baron Of Baby Lamb, Boulangere, Creamed Asparagus Tips (price: 1.5)</li><li>Glazed Sugar Cured Ham With Candied Sweet Potatoes, Whole Spinach (price: 1.1)</li><li>Fried Squab Chicken A La Maryland (1/2) (price: 1.5)</li><li>Petit Tournedao Saute Cocarde (price: 1.75)</li><li>Grilled Kernel Of Sweetbread With Heart Of Artichoke Da Gama (price: 1.5)</li><li>Mushrooms In Crust A La Creme (price: 1.0)</li><li>Small Steak En Casserole Grand'mere (price: 2.25)</li><li>Mushrooms In Crust A La Creme (price: 1.0)</li><li>Vol Au Vent Clementine (price: 1.5)</li><li>Boneless Royal Squab A La Washington (price: 2.0)</li><li>Supreme Of Hot House Chicken Champs Elysees (price: 1.75)</li><li>Roast Beef (price: 1.0)</li><li>Roast Baby Lamb (price: 1.5)</li><li>Roast Royal Squab (price: 1.75)</li><li>Roast Squab Chicken (price: 2.0)</li><li>Broiler (price: 1.5- 3.0)</li><li>Cold Egg, Nicoise Or A La Russe (price: 0.75)</li><li>Cold Egg Finlandaise (price: 0.6)</li><li>Cold Egg With Ham (price: 0.5)</li><li>Egg A La Russe (price: 0.75)</li><li>Cold Halibut Steak A L'espagnole (price: 1.25)</li><li>Cold Kennebec Salmon Parisienne (price: 1.25)</li><li>Cold Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Cold Shrimps A La Joseph (price: 0.75)</li><li>Cold Crab Ravigote (price: 1.0)</li><li>Cold Chicken And Ham Pie (price: 1.5)</li><li>Cold Supreme Plaza (price: 1.75)</li><li>Cold Supreme Jeannette (price: 1.25)</li><li>Cold Head Cheese With Potato Salad (price: 1.0)</li><li>Cold Truffled Galantine Of Capon (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Cold Beef A La Mode In Jelly (price: 1.0)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Cauliflower (price: 0.7)</li><li>Beets On Butter (price: 0.5)</li><li>Jersey Asparagus (price: 0.8)</li><li>Succotash Plaza (price: 0.65)</li><li>Lima Beans (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>New Peas (price: 0.6)</li><li>String Beans (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>White Squash (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Hubbard Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Sweet Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Lettuce, Eggs And Beets Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Orleans Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Tomato Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Hot Semoulina Pudding (price: 0.35)</li><li>Cold Pineapple Pudding (price: 0.35)</li><li>Souffle Martinique (price: 0.75)</li><li>Nicoise Jelly (price: 0.35)</li><li>Crème Caramel (price: 0.3)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Rice Pudding (price: 0.35)</li><li>Strawberry Ice (price: 0.4)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Strawberry Coupe (price: 0.6)</li><li>Pear Cardinal (price: 0.65)</li><li>Orange Sherbet (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Roquefort Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Cocktail Sauce (price: 0.1)</li><li>Little Neck Clams (price: 0.45)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Pineapple Paradise (price: 0.75)</li><li>Melon Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Hors D'oeuvres Varies</li><li>Antipasto (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Cream Soup Ambassadeur (price: 0.5)</li><li>Jellied Consomme Gumbo, Cup (price: 0.4)</li><li>Jellied Consomme Madrilene, Cup (price: 0.4)</li><li>Jellied Consomme Chicken, Cup (price: 0.4)</li><li>Jellied Consomme Beef, Cup (price: 0.4)</li><li>Scrambled Eggs With Mushrooms (price: 0.8)</li><li>Eggs Cocotte Cafe Anglais (price: 0.5)</li><li>Poached Eggs, Comtesse (price: 0.55)</li><li>Omelette Mogador (price: 0.8)</li><li>Grilled Boston Scrod St. Laurent (price: 0.9)</li><li>Boneless Shad And Roe Saute Mascotte (price: 1.25)</li><li>Brook Trout A La Vatel (price: 1.1)</li><li>Filet Of Sole Breval (price: 0.9)</li><li>Soft Shell Crabs Under Glass Maryland (price: 1.25)</li><li>Bluefish Epicurienne (price: 1.1)</li><li>Fried Frogs' Legs With Bacon, Sauce Orly (price: 1.25)</li><li>Supreme Of Sea Bass Florentine (price: 1.1)</li><li>Chicken Halibut Marseillaise (price: 1.1)</li><li>Prime Rib Of Beef With Yorkshire Pudding And New Succotash (price: 1.25)</li><li>Squab Chicken Pot Pie A La Plaza (price: 1.5)</li><li>Grilled Milk Veal Chop, Continental (price: 1.25)</li><li>Noisette Of Spring Lamb A La Paillard (price: 1.5)</li><li>Deerfoot Sausages With Veloute Lyonnaise (price: 0.9)</li><li>Assorted Fresh Vegetables En Bordure Gratinee (price: 1.25)</li><li>Royal Squab En Casserole Beaulieu (price: 2.0)</li><li>Small Steak En Casserole Nivernaise (price: 2.25)</li><li>Baby Lamb Steak Saute Nicoise (price: 2.0)</li><li>Supreme Of Guinea Hen, Metternick (price: 1.75)</li><li>Stuffed Fresh Mushrooms Under Glass Au Beurre D'escargots (price: 1.0)</li><li>Egg, Nicoise (price: 0.75)</li><li>Cold Egg Finlandaise (price: 0.6)</li><li>Cold Egg With Ham (price: 0.5)</li><li>Cold Egg A La Russe (price: 0.75)</li><li>Cold Halibut Steak A L'espagnole (price: 1.25)</li><li>Cold Kennebec Salmon Parisienne (price: 1.25)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Cold Shrimps A La Joseph (price: 0.75)</li><li>Crab Ravigote (price: 1.0)</li><li>Cold Supreme Jeannette (price: 1.25)</li><li>Cold Supreme Plaza (price: 1.75)</li><li>Cold Chicken And Ham Pie (price: 1.5)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Jersey Asparagus (price: 0.8)</li><li>Beets In Butter (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>String Beans (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>New Peas (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>Braised Celery (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Hubbard Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>White Squash (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.6)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Dixie Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Lettuce, Tomato And Cucumber Salad (price: 0.6)</li><li>Kuroki Salad (price: 0.6)</li><li>Louisette Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Romaine Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Portugaise Pudding (price: 0.35)</li><li>Boston Cream Pie (price: 0.35)</li><li>Rhubarb Pie (price: 0.35)</li><li>Baba Au Sirop (price: 0.3)</li><li>Old Fashioned Strawberry Short Cake (price: 0.65)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Mocha Mousse (price: 0.6)</li><li>Pineapple Sherbet (price: 0.4)</li><li>Bar Le Duc Parfait (price: 0.6)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Stilton Cheese (price: 0.6)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Roquefort Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Orange Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Melon Supreme (price: 0.65)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Filet Of Herring (price: 0.5)</li><li>Smoked Salmon (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Hors D'oeuvres Varies</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Antipasto (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>St. Germain Soup (price: 0.4)</li><li>Cream Of Tomato With Rice Soup (price: 0.5)</li><li>Belle Fermiere A La Moelle Soup (price: 0.5)</li><li>Jellied Consomme, Madrilene, Cup (price: 0.4)</li><li>Jellied Consomme, Chicken, Cup (price: 0.4)</li><li>Jellied Consomme, Gumbo, Cup (price: 0.4)</li><li>Jellied Consomme, Beef, Cup (price: 0.4)</li><li>Boiled Live Codfish, Sauce Crevette (price: 0.9)</li><li>Supreme Of Sole Bonne Femme (price: 0.9)</li><li>Boneless Shad And Roe Bretonne (price: 1.25)</li><li>Brook Trout A La Marchand De Vins (price: 1.1)</li><li>Frogs' Legs Rissolees Provencale (price: 1.0)</li><li>Bluefish A La Bartlett (price: 1.1)</li><li>Sea Bass Hawaiian (price: 1.1)</li><li>Lobster In Shell, Thermidor(1/2) (price: 1.5)</li><li>Soft Shell Crabs Meuniere With Bacon And Mushrooms (price: 1.25)</li><li>Boiled Squab Chicken With Rice Pilaw, Sauce Supreme(1/2) (price: 1.5)</li><li>Slice Of Sirloin Of Beef Saute Bristol (price: 1.75)</li><li>Grilled Sweetbread On Toast With Bacon, Mushrooms, And New Succotash (price: 1.5)</li><li>Escalopines Of Milk Veal Au Marsala With Spinach And Noodles In Butter (price: 1.25)</li><li>Glazed Sugar Cured Ham With Vegetables Porte Maillot (price: 1.25)</li><li>Roast Hindquarter Of Baby Lamb, Boulangere (price: 1.5)</li><li>Fresh Mushrooms On Toast With Bacon, Maitre D'hotel (price: 1.0)</li><li>Smothered Hot House Chicken Lathuile(1/2) (price: 1.75)</li><li>Royal Squab A La Turner (price: 2.0)</li><li>Supreme Of Guinea Hen A La Paul With Seedless Grapes (price: 1.75)</li><li>Roast Beef (price: 1.0)</li><li>Roast Baby Lamb (price: 1.5)</li><li>Roast Royal Squab (price: 1.75)</li><li>Broiler(1/2) (price: 1.5- 0.0)</li><li>Roast Squab Chicken (price: 2.0)</li><li>Cold Egg, Nicoise Or A La Russe (price: 0.75)</li><li>Cold Egg With Ham (price: 0.5)</li><li>Cold Egg Findlandaise (price: 0.6)</li><li>Kennebec Salmon Parisienne (price: 1.25)</li><li>Halibut Steak A L'espagnole (price: 1.25)</li><li>Cold Crab Ravigote (price: 1.0)</li><li>Cold Shrimps A La Joseph (price: 0.75)</li><li>Cold Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Cold Supreme Jeannette (price: 1.25)</li><li>Cold Supreme Plaza (price: 1.75)</li><li>Cold Chicken And Ham Pie (price: 1.5)</li><li>Cold Truffled Galantine Of Capon (price: 1.5)</li><li>Cold Head Cheese With Potato Salad (price: 1.0)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Cold Beef A La Mode In Jelly (price: 1.0)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Jersey Asparagus (price: 0.8)</li><li>Cauliflower (price: 0.7)</li><li>Beets In Butter (price: 0.5)</li><li>New Peas (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>String Beans (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>Braised Lettuce (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Hubbard Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Sweet Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Knickerbocker Salad (price: 0.6)</li><li>Lettuce, Orange And Grapefruit Salad (price: 0.6)</li><li>Blackstone Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Rocheford Salad (price: 0.6)</li><li>Endive Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Hot Apricot Pudding (price: 0.35)</li><li>Cold Mogador Pudding (price: 0.35)</li><li>Brandied Pear Jubilee (price: 0.9)</li><li>Souffle Tutti Frutti (price: 0.75)</li><li>Lemon Jelly (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Raspberry Sherbet (price: 0.4)</li><li>Coupe Apricotine (price: 0.6)</li><li>Bombe Reve De Bebe (price: 0.6)</li><li>Camembert Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Roquefort Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Stilton Cheese (price: 0.6)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>King Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Little Neck Clams (price: 0.45)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Fruit Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Melon Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Smoked Salmon (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Hors D'oeuvre Variés</li><li>Stuffed Celery (price: 0.5)</li><li>Anchovy Canape (price: 0.6)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Salami (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Home Made Chicken Soup (price: 0.5)</li><li>Jellied Consommes  Madrilene, Chicken, Beef Or Gumbo, Cup (price: 0.4)</li><li>Omelette Maria (price: 0.8)</li><li>Poached Eggs, Benedict (price: 0.6)</li><li>Scrambled Eggs, St. James (price: 0.8)</li><li>Brook Trout A La Santos Dumont (price: 1.1)</li><li>Supreme Of Sea Bass Lackme (price: 1.1)</li><li>Frogs' Legs Rissolees Lilloise (price: 1.25)</li><li>Soft Shell Crabs Saute With Almonds (price: 1.25)</li><li>Boneless Shad And Roe With Glazed Pineapple (price: 1.25)</li><li>Filet Of Sole Bonnefoy (price: 0.9)</li><li>Kingfish Meuniere A La Bretonne (price: 1.1)</li><li>Baked Lobster Thermidor(1/2) (price: 1.5)</li><li>Grilled Chicken Halibut, Cucumber Vernon (price: 1.1)</li><li>Prime Rib Of Beef, Creamed Potatoes Parisienne, Macedoine Of Vegetables (price: 1.25)</li><li>Grilled Sugar Cured Ham A La Norfolk (price: 1.1)</li><li>Baked Chicken Hash Tettrazini (price: 1.5)</li><li>Kidney Saute Turbigo (price: 1.1)</li><li>Spring Lamb Chop A La Robinson (price: 1.0)</li><li>Milk Veal Cutlet A La Bercy With Veloute Of Cauliflower (price: 1.25)</li><li>Fresh Vegetable Luncheon With Poached Egg (price: 1.25)</li><li>Shoulder Of Baby Lamb En Casserole Nivernaise(For Two) (price: 3.0)</li><li>Tournedo Colbert (price: 2.0)</li><li>Supreme Of Guinea Hen Mikado (price: 1.75)</li><li>Fresh Mushrooms Eugenie (price: 1.0)</li><li>Poussin De Serre En Croute St. Andre (price: 2.25)</li><li>Cold Egg Findlandaise (price: 0.6)</li><li>Egg A La Russe (price: 0.75)</li><li>Cold Egg With Ham (price: 0.5)</li><li>Egg, Nicoise (price: 0.75)</li><li>Cold Halibut Steak A L'espagnole (price: 1.25)</li><li>Cold Kennebec Salmon Parisienne (price: 1.25)</li><li>Cold Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Cold Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Cold Shrimps A La Joseph (price: 0.75)</li><li>Cold Crab Ravigote (price: 1.0)</li><li>Cold Chicken And Ham Pie (price: 1.5)</li><li>Cold Supreme Jeannette (price: 1.25)</li><li>Cold Supreme Plaza (price: 1.75)</li><li>Cold Head Cheese With Potato Salad (price: 1.0)</li><li>Cold Truffled Galantine Of Capon (price: 1.5)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Cold Beef A La Mode In Jelly (price: 1.0)</li><li>Beets In Butter (price: 0.5)</li><li>Jersey Asparagus (price: 0.8)</li><li>Cauliflower (price: 0.7)</li><li>Succotash Plaza (price: 0.65)</li><li>Carrots Vichy (price: 0.5)</li><li>String Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Creamed Spinach (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Voisin Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Sweet Potatoes (price: 0.5)</li><li>Bonnie Brae Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Lettuce, Celery And Pimentos Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Romaine Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Black Cherry Pie (price: 0.35)</li><li>Vanilla Cup Custard (price: 0.25)</li><li>Floating Island, Palmyre (price: 0.4)</li><li>Compote Of Assorted Fruits (price: 0.45)</li><li>Strawberry Tart (price: 0.4)</li><li>French Pastry (1) (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>Strawberry Ice (price: 0.4)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Bar Le Duc Parfait (price: 0.6)</li><li>Mocha Mousse (price: 0.6)</li><li>Pineapple Sherbet (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Berries In Season (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Grapefruit (price: 0.4)</li><li>King Orange (price: 0.2)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Filet Of Herring (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monegasque (price: 0.75)</li><li>Hors D'oeuvre Variés</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Antipasto (price: 0.5)</li><li>Salami (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Tomato Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Essence De Volaille Theodora (price: 0.5)</li><li>Home Made Chicken Soup (price: 0.5)</li><li>Jellied Consommes  Madrilene, Chicken, Beef Or Gumbo, Cup (price: 0.4)</li><li>Planked Boneless Shad And Roe Lady Duveene (price: 1.25)</li><li>Brook Trout Au Bleu, Sauce Escoffier (price: 1.0)</li><li>Filet Of Sole Cafe De Paris (price: 0.9)</li><li>Frogs' Legs A La Cavour (price: 1.25)</li><li>Baked Lobster Beaugency(1/2) (price: 1.5)</li><li>Grilled Bluefish Baltimore (price: 1.1)</li><li>Supreme Of Kingfish Fin De Siecle (price: 1.1)</li><li>Soft Shell Crabs Under Glass Archiduc (price: 1.25)</li><li>Sea Bass Saute Nicoise (price: 1.1)</li><li>Sea Bass Saute Nicoise (price: 1.1)</li><li>Smothered Squab Chicken A La Dickens With Asparagus Tips(1/2) (price: 1.5)</li><li>Roast Hindquarter Of Baby Lamb, Potatoes Parisienne, Succotash Virginia (price: 1.5)</li><li>Glazed Virginia Ham, Sauce Porto With Sweet Potatoes Dixie, Braised Endives (price: 1.25)</li><li>Grilled Kernel Of Sweetbread With Bacon And Heart Of Artichoke Princesse (price: 1.5)</li><li>Escalopines Of Milk Veal Au Vin Blanc With String Beans And Veloute Crecy (price: 1.25)</li><li>Larded Tenderloin Of Beef, Richelieu (price: 2.0)</li><li>Planked Porterhouse Steak Aux Primeurs (price: 3.0)</li><li>Imperial Squab Souvaroff (price: 2.0)</li><li>Patty Of Mushrooms A La Reine (price: 1.0)</li><li>Supreme Of Guinea Hen Tour D'argent (price: 1.75)</li><li>Grilled Egg Plant And Tomato Baltimore (price: 0.9)</li><li>Hot House Chicken L'athuile(1/2) (price: 1.75)</li><li>Roast Beef (price: 1.0)</li><li>Baby Lamb (price: 1.5)</li><li>Roast Royal Squab (price: 1.75)</li><li>Broiler Chicken (price: 1.5012- 0.0)</li><li>Squab Chicken (price: 2.0)</li><li>Egg, Nicoise (price: 0.75)</li><li>Egg A La Russe (price: 0.75)</li><li>Cold Egg Findlandaise (price: 0.6)</li><li>Cold Egg With Ham (price: 0.5)</li><li>Cold Kennebec Salmon Parisienne (price: 1.25)</li><li>Cold Halibut Steak A L'espagnole (price: 1.25)</li><li>Cold Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Cold Crab Ravigote (price: 1.0)</li><li>Cold Shrimps A La Joseph (price: 0.75)</li><li>Cold Chicken And Ham Pie (price: 1.5)</li><li>Cold Supreme Plaza (price: 1.75)</li><li>Cold Supreme Jeannette (price: 1.25)</li><li>Cold Truffled Galantine Of Capon (price: 1.5)</li><li>Cold Head Cheese With Potato Salad (price: 1.0)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Cold Beef A La Mode In Jelly (price: 1.0)</li><li>Jersey Asparagus (price: 0.8)</li><li>Cauliflower (price: 0.7)</li><li>Beets In Butter (price: 0.5)</li><li>Carrots Vichy (price: 0.5)</li><li>String Beans (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>New Peas (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Creamed Spinach (price: 0.6)</li><li>Hubbard Squash (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Fried Egg Plant (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Potatoes, Dixie (price: 0.5)</li><li>Potatoes Biron (price: 0.45)</li><li>Sweet Potatoes (price: 0.5)</li><li>Potatoes Voisin (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Potatoes, Plaza (price: 0.5)</li><li>Salad Louise (price: 0.6)</li><li>Salad Richard (price: 0.6)</li><li>Lettuce, Cucumber And Beets Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Salad, Romaine (price: 0.45)</li><li>Salad, Mixed Green (price: 0.45)</li><li>Salad, Endive (price: 0.45)</li><li>Salad, (price: 0.45)</li><li>Salad, Lettuce (price: 0.45)</li><li>Salad, Tomato (price: 0.45)</li><li>Cold Mocha Custard Pudding (price: 0.35)</li><li>Cold Mocha Custard Pudding (price: 0.35)</li><li>Hot Caroline Pudding (price: 0.35)</li><li>Figs A La Kimberley (price: 0.65)</li><li>Bavarian Cream Aveline (price: 0.4)</li><li>Souffle Africaine (price: 0.75)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry(1) (price: 0.25)</li><li>Rice Pudding (price: 0.35)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Bombe Reve De Bebe (price: 0.6)</li><li>Coupe Apricotine (price: 0.6)</li><li>Raspberry Sherbert (price: 0.4)</li><li>Port Du Salut (price: 0.45)</li><li>Brie (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Bel Paese (price: 0.45)</li><li>Camembert (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Liederkranz Cheese</li><li>Stilton Cheese (price: 0.6)</li><li>Roquefort (price: 0.5)</li><li>Gorgonzola (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Berries In Season (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Grapes (price: 0.5)</li><li>King Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Pear (price: 0.2)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Fruit Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Filet Of Herring (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Smoked Salmon (price: 0.75)</li><li>Hors D'oeuvre Varies</li><li>Stuffed Celery (price: 0.5)</li><li>Anchovy Canape (price: 0.6)</li><li>Westphalia Ham (price: 1.25)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Cream Of White Beans Soup Beauharnaise (price: 0.5)</li><li>Jellied Consomme, Chicken (price: 0.4)</li><li>Jellied Consomme, Madrilene (price: 0.4)</li><li>Gumbo (price: 0.4)</li><li>Jellied Consomme, Beef (price: 0.4)</li><li>Scrambled Eggs, Grand' Mere (price: 0.8)</li><li>Poached Eggs, Armenonville (price: 0.55)</li><li>Omelette With Lamb Kidney, Madeira (price: 0.8)</li><li>Creamed Salmon In Bordure Florentine (price: 1.25)</li><li>Chicken Halibut, Los Angeles (price: 1.1)</li><li>Filet Of Sea Bass, Lady Egmont (price: 1.1)</li><li>Frogs' Legs Rissolees Royer (price: 1.25)</li><li>Supreme Of Sole Commodore (price: 0.9)</li><li>Boneless Shad And Roe Beaucaire (price: 1.25)</li><li>Brook Trout A La Danicheff (price: 1.1)</li><li>Grilled Bluefish, Jardiniere (price: 1.1)</li><li>Kingfish, Amelie (price: 1.1)</li><li>Fried Soft Shell Crabs With Bacon, Sauce Remoulade (price: 1.25)</li><li>Deviled Slice Of Roast Beef With Bone, Fried Sweet Potatoes, Mixed Beans (price: 1.25)</li><li>Chicken Patty A La Reine (price: 1.25)</li><li>Roast Rack Of Milk Veal Bouulangere (price: 1.5)</li><li>Sugar Cured Ham Saute With Glazed Apple And New Succotash (price: 1.1)</li><li>Fresh Vegetable Luncheon Au Gratin With Poached Egg (price: 1.25)</li><li>Epigramme Of Baby Lamb With Spaghetti Caruso (price: 1.25)</li><li>Supreme Of Guinea Hen Rossini (price: 1.75)</li><li>Imperial Squab A L'americaine (price: 2.0)</li><li>Baked Fresh Mushrooms Bourguignonne (price: 1.0)</li><li>Chicken Saute Lafayette (1/2) (price: 1.75)</li><li>Egg With Ham (price: 0.5)</li><li>Egg Finlandaise (price: 0.6)</li><li>Egg A La Russe (price: 0.75)</li><li>Egg, Nicoise (price: 0.75)</li><li>Halibut Steak A L'espagnole (price: 1.25)</li><li>Kennebec Salmon Parisienne (price: 1.25)</li><li>Shrimps A La Joseph (price: 0.75)</li><li>Crab Ravigote (price: 1.0)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Supreme Plaza (price: 1.75)</li><li>Chicken And Ham Pie (price: 1.5)</li><li>Supreme Jeannette (price: 1.25)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Cauliflower (price: 0.7)</li><li>Beets In Butter (price: 0.5)</li><li>Jersey Asparagus (price: 0.8)</li><li>Carrots Vichy (price: 0.5)</li><li>Succotash Plaza (price: 0.65)</li><li>String Beans (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>White Squash (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Hubbard Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Boiled Bermuda (price: 0.3)</li><li>Voisin (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Biron (price: 0.45)</li><li>Dixie Potatoes (price: 0.5)</li><li>Sunrise Salad (price: 0.6)</li><li>Lily Salad (price: 0.6)</li><li>Princess Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Salad, Plaza Special (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Rocheford Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Mixed Green Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Florida Pudding (price: 0.35)</li><li>Lemon Custard Pie (price: 0.35)</li><li>Apricot Pie (price: 0.35)</li><li>Mocha Layer Cake (price: 0.4)</li><li>Almond Cream Puff (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Strawberry Ice (price: 0.4)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Bar Le Duc Parfait (price: 0.6)</li><li>Pineapple Sherbet (price: 0.4)</li><li>Mocha Mousse (price: 0.6)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut (price: 0.45)</li><li>Bel Paese (price: 0.45)</li><li>Camembert (price: 0.45)</li><li>Brie (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Stilton (price: 0.6)</li><li>Roquefort (price: 0.5)</li><li>Gorgonzola (price: 0.5)</li><li>Berries In Season (price: 0.6)</li><li>Watermelon (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Little Neck Clams (price: 0.45)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Orange Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Filet Of Herring (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Hors D'oeuvre Varies</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Chicken Okra Soup (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Tomato Soup (price: 0.4)</li><li>Pot Au Feu A La Moelle (price: 0.5)</li><li>Cream Of New Peas Soup A La Hugo (price: 0.6)</li><li>Jellied Consomme, Chicken (price: 0.4)</li><li>Jellied Consomme, Beef (price: 0.4)</li><li>Gumbo (price: 0.4)</li><li>Jellied Consomme, Madrilene (price: 0.4)</li><li>Poached Kennebec Salmon, Sauce Joinville (price: 1.1)</li><li>Supreme Of Sole Duchesse (price: 0.9)</li><li>Soft Shell Crabs Rissolees With Almonds (price: 1.25)</li><li>Chicken Halibut à La Turque (price: 1.1)</li><li>Boneless Shad And Roe Belle Meuniere (price: 1.25)</li><li>Grilled Bluefish Caprice (price: 1.1)</li><li>Sea Bass Sauté Alphonse Xiii (price: 1.1)</li><li>Baked Lobster In Shell Majestic (1/2) (price: 1.5)</li><li>Brook Trout A La Canajoharie (price: 1.1)</li><li>Patty Of Frogs' Legs Poulette (price: 1.25)</li><li>Squab Chicken En Casserole With Mushrooms And Fresh Vegetables (1/2) (price: 1.5)</li><li>Escalopines Of Milk Veal A L'anglaise With Whole Spinach And Carrots Vichy (price: 1.25)</li><li>Larded Kernel Of Sweetbread Demidoff With Potatoes Lorette (price: 1.5)</li><li>Grilled Loin Of Baby Lamb With Kidney, Nicoise (price: 1.5)</li><li>Sliced Filet Of Beef, Armenonville (price: 1.75)</li><li>Grilled Fresh Mushrooms On Toast With Ham Au Beurre D'escargots (price: 1.0)</li><li>Chicken Saute Frou Frou (1/2) (price: 1.75)</li><li>Imperial Squab En Compote (price: 2.0)</li><li>Quenelles Of Ham Florentine (price: 1.1)</li><li>Noisette Of Spring Lamb Lucullus (price: 1.5)</li><li>Roast Beef (price: 1.0)</li><li>Roast Baby Lamb (price: 1.5)</li><li>Roast Royal Squab (price: 1.75)</li><li>Broiler (price: 3.0)</li><li>Broiler (1/2) (price: 1.5)</li><li>Roast Squab Chicken (price: 2.0)</li><li>Egg With Ham (price: 0.5)</li><li>Egg Finlandaise (price: 0.6)</li><li>Egg A La Russe (price: 0.75)</li><li>Egg, Nicoise (price: 0.75)</li><li>Kennebec Salmon Parisienne (price: 1.25)</li><li>Halibut Steak A L'espagnole (price: 1.25)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Shrimps A La Joseph (price: 0.75)</li><li>Crab Ravigote (price: 1.0)</li><li>Supreme Jeannette (price: 1.25)</li><li>Supreme Plaza (price: 1.75)</li><li>Chicken And Ham Pie (price: 1.5)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Jersey Asparagus (price: 0.8)</li><li>Cauliflower (price: 0.7)</li><li>Beets In Butter (price: 0.5)</li><li>Lima Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>String Beans (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>Succotash Plaza (price: 0.65)</li><li>Braised Lettuce (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Creamed Spinach (price: 0.6)</li><li>Hubbard Squash (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Sweet Potatoes (price: 0.5)</li><li>Dixie Salad (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Plaza Potatoes (price: 0.5)</li><li>Jeannette Salad (price: 0.6)</li><li>Hot Bresilien Pudding (price: 0.35)</li><li>Cold Cocoanut Pudding (price: 0.35)</li><li>Mont Blanc Aux Marrons (price: 0.4)</li><li>Souffle Duchesse (price: 0.75)</li><li>Orange Jelly (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Ice Cream, Caramel (price: 0.45)</li><li>Ice Cream, Fresh Strawberry (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Raspberry Sherbet (price: 0.4)</li><li>Coupe Apricotine (price: 0.6)</li><li>Bombe Reve De Bebe (price: 0.6)</li><li>Brie (price: 0.45)</li><li>Camembert (price: 0.45)</li><li>Port Du Salut (price: 0.45)</li><li>Bel Paese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Stilton Cheese (price: 0.6)</li><li>Gorgonzola (price: 0.5)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Cantaloupe (price: 0.6)</li><li>Watermelon (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Berries In Season (price: 0.6)</li><li>King Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Orange (price: 0.2)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Pineapple Paradise (price: 0.75)</li><li>Fresh Caviar (price: 2.25)</li><li>Filet Of Herring (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Calf's Feet Soup, English Style (price: 0.5)</li><li>Noisette Of Mutton Chantilly (price: 1.25)</li><li>Egg With Ham (price: 0.5)</li><li>Huckleberry Pie (price: 0.35)</li><li>Dinner</li><li>Clams And Cocktails</li><li>Little Neck Clams (price: 0.45)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Melon Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Filet Of Herring (price: 0.5)</li><li>Smoked Salmon (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Hors D'oeuvre Variés</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Macquereau Au Vin Blanc (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Cream Of Mushrooms Aux Souffles Soup (price: 0.5)</li><li>Boula Gratinee Soup (price: 0.5)</li><li>Chicken, Jellied Consomme, Cup (price: 0.4)</li><li>Beef, Jellied Consomme, Cup (price: 0.4)</li><li>Gumbo, Cup (price: 0.4)</li><li>Madrilene, Jellied Consomme, Cup (price: 0.4)</li><li>Boiled Chicken Halibut, Sauce Mousseline (price: 1.1)</li><li>Soft Shell Crabs A La Creole (price: 1.25)</li><li>Baked Lobster Grand Hotel (1/2) (price: 1.5)</li><li>Supreme Of Sole Victoria (price: 0.9)</li><li>Turban Of Frogs' Legs Regence (price: 1.5)</li><li>Brook Trout Matelotte (price: 1.1)</li><li>Grilled Bluefish With Bacon Au Capeco (price: 1.1)</li><li>Boneless Shad And Roe A L'orange (price: 1.25)</li><li>Darne Of Salmon Marshall Joffre (price: 1.25)</li><li>Slice Of Beef Tenderloin Massenet (price: 1.75)</li><li>Supreme Of Hot House Chicken A La Paul With Seedless Grapes (price: 1.75)</li><li>Grenadin Of Milk Veal Lavalliere With Asparagus Tips (price: 1.25)</li><li>Roast Rack Of Spring Lamb, Boulangere (price: 1.5)</li><li>Glazed Virginia Ham With Mashed Yams, Creamed Spinach, Sauce Porto (price: 1.25)</li><li>Grilled Kernel Of Sweetbread With Mushrooms And String Beans Polonaise (price: 1.5)</li><li>Mushrooms On Toast With Bacon, Beurre Ravigotte (price: 1.0)</li><li>Planked American Steak A La Plaza (price: 3.0)</li><li>Boneless Royal Squab A La Turner (price: 2.0)</li><li>Baby Lamb Steak Saute Bonne Femme (price: 2.0)</li><li>Arroz Con Pollo A L'espagnole (1/2) (price: 1.75)</li><li>Roast Beef (price: 1.0)</li><li>Baby Lamb (price: 1.5)</li><li>Royal Squab (price: 1.75)</li><li>Squab Chicken (price: 2.0)</li><li>Broiler To Order (price: 3.0)</li><li>Broiler (1/2) (price: 1.5)</li><li>Egg With Ham (price: 0.5)</li><li>Egg, Nicoise (price: 0.75)</li><li>Egg, A La Russe (price: 0.75)</li><li>Egg Finlandaise (price: 0.6)</li><li>Kennebec Salmon Parisienne (price: 1.25)</li><li>Halibut Steak A L'espagnole (price: 1.25)</li><li>Chicken Lobster, Mayonnaise (price: 1.5)</li><li>Crab Ravigote (price: 1.0)</li><li>Shrimps A La Joseph (price: 0.75)</li><li>Supreme Jeannette (price: 1.25)</li><li>Chicken And Ham Pie (price: 1.5)</li><li>Chicken And Ham Pie (price: 1.5)</li><li>Supreme Plaza (price: 1.75)</li><li>Truffled Galantine Of Capon (price: 1.5)</li><li>Head Cheese With Potato Salad (price: 1.0)</li><li>Assorted Cold Cuts (price: 1.5)</li><li>Beef A La Mode In Jelly (price: 1.0)</li><li>Assorted Cold Cuts With Chicken (price: 1.75)</li><li>Cauliflower (price: 0.7)</li><li>Jersey Asparagus (price: 0.8)</li><li>Beets In Butter (price: 0.5)</li><li>Carrots Vichy (price: 0.5)</li><li>Lima Beans (price: 0.6)</li><li>String Beans (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>New Peas.6</li><li>Stewed Okra (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>White Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Hubbard (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Biron Potatoes (price: 0.45)</li><li>Sweet Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Voisin Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Excelsior Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Royal Salad (price: 0.6)</li><li>Juliette Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Romaine Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Hot Souveraine Pudding (price: 0.35)</li><li>Cold Cottage Pudding (price: 0.35)</li><li>Rice Imperatrice (price: 0.4)</li><li>Wild Cherry Jelly (price: 0.35)</li><li>Souffle Tosca (price: 0.75)</li><li>French Pastry (1) (price: 0.25)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Raspberry Sherbet (price: 0.4)</li><li>Bombe Reve De Bebe (price: 0.6)</li><li>Coupe Apricotine (price: 0.6)</li><li>Camembert Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Roquefort Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Watermelon (price: 0.6)</li><li>Berries In Season (price: 0.6)</li><li>Orange (price: 0.2)</li><li>Pear (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Orange Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Melon Supreme (price: 0.65)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monegasque (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Hors D'oeuvre Variés</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Macquereau Au Vin Blanc (price: 0.5)</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Cream Germiny (price: 0.5)</li><li>Crab Meat Gumbo (price: 0.5)</li><li>Chicken, Jellied Consomme, Cup (price: 0.4)</li><li>Madrilene, Jellied Consomme, Cup (price: 0.4)</li><li>Gumbo, Cup (price: 0.4)</li><li>Beef, Jellied Consomme, Cup (price: 0.4)</li><li>Poached Eggs, Benedict (price: 0.6)</li><li>Eggs, Shirred A La Turque (price: 0.5)</li><li>Omelette Suissesse (price: 0.8)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Tomato Soup (price: 0.4)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Pineapple Paradise (price: 0.75)</li><li>Melon Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monegasque (price: 0.75)</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Hors D'oeuvres Varies</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>St. Germain Soup (price: 0.4)</li><li>Onion Soup Gratine (price: 0.5)</li><li>Julienne Darblay Soup (price: 0.5)</li><li>Cocotte Eggs, Josephine (price: 0.5)</li><li>Minced Eggs A La Poulette (price: 0.75)</li><li>Poached Eggs, Bonne Femme (price: 0.55)</li><li>Omelette, Salvator (price: 0.8)</li><li>Scrambled Eggs, Rachel (price: 0.8)</li><li>Fried Soft Shell Crabs With Bacon, Sauce Remoulade (price: 1.25)</li><li>Creamed Finnan Haddie A La Douglas (price: 0.9)</li><li>Chicken Halibut, Los Angeles (price: 1.1)</li><li>Darne Of Salmon Flaubert (price: 1.25)</li><li>Frogs' Legs Rissolees Royer (price: 1.25)</li><li>Filet Of Sea Bass Lady Egmont (price: 1.1)</li><li>Stuffed Smelts A La Tailer (price: 1.1)</li><li>Grilled Bluefish, Jardiniere (price: 1.1)</li><li>Brook Trout A La Dussolier (price: 1.1)</li><li>Boneless Shad And Roe Beaucaire (price: 1.25)</li><li>Supreme Of Sole Commodore (price: 0.9)</li><li>Deviled Breast Of Baby Lamb Jardiniere With Veloute Parmentier (price: 1.25)</li><li>Ragout A La Deutsch (price: 1.25)</li><li>Grilled Squab Chicke Continental (1/2) (price: 1.5)</li><li>Browned Corned Beef Hash With Fried Egg And Carrots Vichy (price: 0.9)</li><li>Duck Livers Saute Bordelaise In Bordure Of Rice (price: 0.9)</li><li>Roast Rack Of Milk Veal Boulangere (price: 1.25)</li><li>Fresh Vegetable Luncheon Au Gratin (price: 1.25)</li><li>Gnocchi Romaine (price: 0.5)</li><li>Macaroni Caruso (price: 0.9)</li><li>Spaghetti A La Milanaise (price: 0.75)</li><li>Smothered Ham Steak Southern Style (price: 1.25)</li><li>Noisette De Pre Sale A La Paillard (price: 1.5)</li><li>Supreme Of Guinea Hen Rossini (price: 1.75)</li><li>Imperial Squab A L'americaine (price: 2.0)</li><li>Chicken Saute Lafayette (1/2) (price: 1.75)</li><li>Fresh Mushrooms On Toast With Bacon, Beurre D'escargots (price: 1.0)</li><li>Small Steak En Casserole Chez Soi (price: 2.25)</li><li>California Asparagus (price: 0.9)</li><li>Beets In Butter (price: 0.5)</li><li>Stewed Okra (price: 0.6)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>Carrots Vichy (price: 0.5)</li><li>New Peas (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>String Beas (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>Braised Celery (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Parsnips, Fines Herbes (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>Fried Egg Plant (price: 0.5)</li><li>Oyster Plant, Poulette (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Bennet Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Jeannette Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Mimosa Salad (price: 0.6)</li><li>Lettuce, Tomato And Asparagus Tips Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Dandelion Salad A L'allemande (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Lettuce Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Diplomate Pudding (price: 0.35)</li><li>Caramel Custard Pie (price: 0.35)</li><li>Red Sour Cherry Pie (price: 0.35)</li><li>Mocha Layer Cake (price: 0.4)</li><li>Apple Tart (price: 0.4)</li><li>Rice Pudding (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Mint Sherbet (price: 0.4)</li><li>Tutti Frutti Parfait (price: 0.6)</li><li>Mocha Mousse (price: 0.6)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Roquefort Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Stilton Cheese (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Delicious Apple (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>King Orange (price: 0.2)</li><li>Little Neck Clams (price: 0.45)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Cream Of Golden Bantam Corn Savarin (price: 0.5)</li><li>Chicken Printaniere With Barley (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Plaza Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Voisin Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Hartford Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Princess Salad (price: 0.6)</li><li>Lily Salad (price: 0.6)</li><li>Plaza Special Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Dandelion Salad A L'allemande (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Escarole Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Hot Tutti Frutti Pudding (price: 0.35)</li><li>Cold Pralinee Custard Pudding (price: 0.35)</li><li>Rice Imperatrice (price: 0.4)</li><li>Souffle Montreuil (price: 0.75)</li><li>Crepes Suzette (price: 1.0)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Strawberry Coupe (price: 0.6)</li><li>Orange Sherbet (price: 0.4)</li><li>Bombe Nelusko (price: 0.6)</li><li>Camembert Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Grapefruit (price: 0.4)</li><li>King Orange (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Cocktail Sauce (price: 0.1)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Pineapple Paradise (price: 0.75)</li><li>Orange, Grapefruit, Melon Or Fruit Supreme (price: 0.65)</li><li>Fresh Caviar (price: 2.25)</li><li>Soft Shell Crabs On Toast With Bacon, Beurre Ravigotte (price: 1.25)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Pineapple Paradise (price: 0.75)</li><li>Orange, Grapefruit, Melon Or Fruit Supreme (price: 0.65)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Tomato Monegasque (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Stuffed Celery (price: 0.5)</li><li>Anchovy Canape (price: 0.6)</li><li>Hors D'oeuvre Variés</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>St. Germain Soup (price: 0.4)</li><li>Ox Tail Soup Parisienne (price: 0.5)</li><li>Cream Of Asparagus Soup Aux Quenelles (price: 0.5)</li><li>Soft Shell Crabs A La Creole (price: 1.25)</li><li>Boiled Chicken Halibut, Sauce Mousseline (price: 1.1)</li><li>Boneless Shad And Roe Mascotte (price: 1.25)</li><li>Filet Of Sea Bass, Miss Poulette (price: 1.1)</li><li>Baked Lobster Grand Hotel (1/2) (price: 1.5)</li><li>Supreme Of Sole Suzette (price: 0.9)</li><li>Brook Trout Matelotte (price: 1.1)</li><li>Grilled Bluefish With Bacon Au Capeco (price: 1.1)</li><li>Darne Of Salmon Montgolfier (price: 1.25)</li><li>Aiguillette Of Kingfish Trianon (price: 1.1)</li><li>Turban Of Frogs' Legs And Shrimps A L'americaine (price: 1.25)</li><li>Roast Baron Of Baby Lamb Boulangere With Mixed Beans Forestiere (price: 1.5)</li><li>Tournedo Saute Cocarde (price: 2.0)</li><li>Fricassee Of Half Squab Chicken Normande (price: 1.5)</li><li>Grilled Kernel Of Sweetbread With Ham, Tomato, Mushrooms, Green Pepper (price: 1.5)</li><li>Braised Loin Of Milk Veal Bourgeoise With Gnocchi Romaine (price: 1.25)</li><li>Calf's Head A L'huile (price: 0.9)</li><li>Grilled Mushrooms On Toast With Bacon, Beurre Ravigotte (price: 1.0)</li><li>Boneless Royal Squab A La Turner (price: 2.0)</li><li>Planked American Steak A La Plaza (price: 3.0)</li><li>Mousseline Of Ham National (price: 1.1)</li><li>Arroz Con Pollo A L'espagnole (1/2) (price: 1.75)</li><li>Vol Au Vent Of Green Turtle Madiera (price: 1.5)</li><li>Supreme Of Chicken Favorite (price: 1.75)</li><li>Noisette De Pre Sale A L'estragon With Cucumber A L'etuve (price: 1.5)</li><li>Galantine Of Capon Edward Vii (price: 1.5)</li><li>Baby Lamb (price: 1.5)</li><li>Roast Beef (price: 1.0)</li><li>Squab Chicken (price: 2.0)</li><li>Broiler (price: 3.0)</li><li>Broiler (1/2) (price: 1.5)</li><li>Royal Squab (price: 1.75)</li><li>Beets In Butter (price: 0.5)</li><li>California Asparagus (price: 0.9)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>Stewed Okra (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>New Peas, String Beans Or Lima Beans (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>Artichoke (price: 0.6)</li><li>Braised Lettuce Or Celery (price: 0.6)</li><li>Parsnips, Fines Herbes (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Hubbard Or White Squash (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Oyster Plant, Poulette (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Sweet Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Plaza Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Blue Points (price: 0.45)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Cape Cods (price: 0.5)</li><li>Cotuits (price: 0.5)</li><li>Blue Points (price: 0.45)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Melon Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Orange Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Fresh Caviar (price: 2.25)</li><li>Clams, Blue Points (price: 0.45)</li><li>Clams, Cape Cods (price: 0.5)</li><li>Clams, Cotuits (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaze (price: 1.0)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Pineapple Paradise (price: 0.75)</li><li>Melon Supreme (price: 0.65)</li><li>Orange Supreme (price: 0.65)</li><li>Grapefruit Supreme (price: 0.65)</li><li>Fruit Supreme (price: 0.65)</li><li>Philadelphia Pepper Pot (price: 0.5)</li><li>Eggs Cocotte, Josephine (price: 0.5)</li><li>Poached Eggs, Bonne Femme (price: 0.55)</li><li>Eggs, Minced A La Poulette (price: 0.75)</li><li>Omelette, Agnes Sorel (price: 0.8)</li><li>Scrambled Eggs, Grand' Mere (price: 0.8)</li><li>Deviled Breast Of Baby Lamb With Brains Jardiniere With Veloute Parmentier (price: 1.25)</li><li>Milk Veal Cutlet Hongroise With Noodles (price: 1.25)</li><li>Grilled Squab Chicken Continental (1/2) (price: 1.5)</li><li>Braised Ham Steak, Southern Style (price: 1.25)</li><li>Knickerbocker Salad (price: 0.6)</li><li>Milady Pudding (price: 0.35)</li><li>Brie Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Stilton Cheese (price: 0.6)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Grapefruit (price: 0.4)</li><li>Orange (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>King Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Blue Points (price: 0.45)</li><li>Pineapple Paradise (price: 0.75)</li><li>Orange, Grapefruit, Melon Or Fruit Supreme (price: 0.65)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Plaza Appetizer (price: 1.25)</li><li>Tomato Monegasque (price: 0.75)</li><li>Smoked Salmon (price: 0.75)</li><li>Hors D'oeuvre Variés</li><li>Stuffed Celery (price: 0.5)</li><li>Anchovy Canape (price: 0.6)</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>St. Germain (price: 0.4)</li><li>Calf's Feet Soup A L'anglaise (price: 0.5)</li><li>Cream Of Yelllow Squash Soup Aux Croutons (price: 0.5)</li><li>Quenelle Of Capon, Sauce Archiduc With Purée Of Peas (price: 1.25)</li><li>Poussin De Serre A La Souvaroff (price: 2.25)</li><li>Boneless Royal Squab A La Washington (price: 2.0)</li><li>Filet Mignon Béatrice (price: 2.0)</li><li>Supreme Of Hot House Chicken Rossini (price: 1.75)</li><li>Turban De Ris D'agneau Bonne Maman (price: 1.25)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Little Neck Clams (price: 0.45)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>California Asparagus (price: 0.9)</li><li>Beets In Butter (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>Stewed Okra (price: 0.6)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Carrots Vichy (price: 0.5)</li><li>Succotash Plaza (price: 0.65)</li><li>Parsnips, Fines Herbes (price: 0.5)</li><li>Artichoke (price: 0.6)</li><li>Creamed Spinach (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Oyster Plant, Poulette (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Sweet Onions In Cream (price: 0.6)</li><li>Biron Potatoes (price: 0.45)</li><li>Plaza Potatoes (price: 0.5)</li><li>Bennet Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Princess Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Lily Salad (price: 0.6)</li><li>Plaza Special Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Dandelion A L'allemande Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Romaine Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Boston Cream Pie (price: 0.35)</li><li>Portugaise Pudding (price: 0.35)</li><li>Rhubarb Pie (price: 0.35)</li><li>Baba Au Sirop (price: 0.3)</li><li>Old Fashioned Strawberry Short Cake (price: 0.65)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Rice Pudding (price: 0.35)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Grenadine Sherbet (price: 0.4)</li><li>Strawberry Mousse (price: 0.6)</li><li>Parfait Praline (price: 0.6)</li><li>Camembert Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Roquefort Cheese (price: 0.5)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>King Orange (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Little Neck Clams (price: 0.45)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Orange (price: 0.65)</li><li>Pineapple Paradise (price: 0.75)</li><li>Fruit Supreme (price: 0.65)</li><li>Grapefruit (price: 0.65)</li><li>Melon (price: 0.65)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Filet Of Herring (price: 0.5)</li><li>Fresh Caviar (price: 2.25)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monegasque (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Hors D'oeuvre Variés</li><li>Stuffed Celery (price: 0.5)</li><li>Anchovy Canape (price: 0.6)</li><li>Salami (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Antipasto (price: 0.5)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Chicken Okra (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>Home Made Chicken Soup (price: 0.5)</li><li>Clear Mock Turtle (price: 0.5)</li><li>Scrambled Eggs, Chasseur (price: 0.8)</li><li>Eggs Shirred, Meyerbeer (price: 0.5)</li><li>Poached, Benedict (price: 0.6)</li><li>Omelette Maria (price: 0.8)</li><li>Cocotte, Bergere (price: 0.5)</li><li>Supreme Of Sea Bass Florentine (price: 1.1)</li><li>Brook Trout Hoteliere (price: 1.1)</li><li>Frog's Legs Rissolees Provencale (price: 1.0)</li><li>Soft Shell Crabs Saute With Almonds (price: 1.25)</li><li>Filet Of Sole Veronique (price: 0.9)</li><li>Boneless Shad And Roe With Glazed Cantaloupe (price: 1.25)</li><li>Chicken Halibut, Cucumber Vernon (price: 1.1)</li><li>Kingfish Meuniere A La Bretonne (price: 1.1)</li><li>Grilled Bluefish With Bacon And Mushrooms (price: 1.1)</li><li>Cold Salmon Parisienne (price: 1.25)</li><li>Baked Lobster Thermidor (1/2) (price: 1.5)</li><li>Prime Rib Of Beef With Potatoes Maitre D'hotel And Stuffed Green Peppers (price: 1.25)</li><li>Grilled Sugar Cured Ham With Risotto Plaza (price: 1.1)</li><li>Baked Shell Of Capon St. Germain (price: 1.5)</li><li>Milk Veal Cutlet A La Viennoise (price: 1.25)</li><li>Kidney Saute Berrichonne (price: 1.1)</li><li>Spring Lamb Chop A La Pompadour (price: 1.0)</li><li>Fresh Vegetable Luncheon With Poached Egg (price: 1.25)</li><li>Spaghetti A La Milanaise (price: 0.75)</li><li>Gnocchi Romaine (price: 0.5)</li><li>Macaroni Caruso (price: 0.9)</li><li>Shoulder Of Baby Lamb En Casserole Grand'mere (For Two) (price: 3.0)</li><li>Fresh Mushrooms Under Glass New Orleans (price: 1.0)</li><li>Hot House Chicken A La Biarritz (1/2) (price: 1.75)</li><li>Duck Livers A La Teddy (price: 0.9)</li><li>Tournedo A La Morland (price: 2.0)</li><li>Supreme Of Guinea Hen Mikado (price: 1.75)</li><li>Poussin De Serre En Croute St. Genis (price: 2.25)</li><li>Brizola Steak Tyrolienne (price: 1.75)</li><li>Cassolette Of Sweetbread Hongroise (price: 1.25)</li><li>Beets In Butter (price: 0.5)</li><li>California Asparagus (price: 0.9)</li><li>Cauliflower (price: 0.7)</li><li>Stewed Okra (price: 0.6)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>New Peas (price: 0.6)</li><li>String Beans (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>Carrots Vichy (price: 0.5)</li><li>Lima Beans (price: 0.6)</li><li>Parsnips, Fines Herbes (price: 0.5)</li><li>Braised Lettuce (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Oyster Plant, Poulette (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Small Onions In Cream (price: 0.6)</li><li>Sweet Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Voisin Potatoes (price: 0.5)</li><li>Bennet Potatoes (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Lettuce, Eggs And Beets Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Orleans Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Dandelion Salad A L'allemande (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Escarole Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Mocha Cup Custard (price: 0.25)</li><li>Peach Pie (price: 0.35)</li><li>Floating Island Duchesse (price: 0.4)</li><li>Compote Of Assorted Fruits (price: 0.45)</li><li>Strawberry Tart (price: 0.4)</li><li>French Pastry (1) (price: 0.25)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Strawberry Mousse (price: 0.6)</li><li>Parfait Praline (price: 0.6)</li><li>Grenadine Sherbet (price: 0.4)</li><li>Camembert Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Stilton Cheese (price: 0.6)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Roquefort Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>King Orange (price: 0.2)</li><li>Dixie Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Plaza Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Excelsior Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Juliette Salad (price: 0.6)</li><li>Royal Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Dandelion Salad A L'allemande (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Endive Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Hot Royal Pudding (price: 0.35)</li><li>Cold Fig Pudding (price: 0.35)</li><li>Vanilla Souffle, Sabayon Sauce (price: 0.75)</li><li>Profiteroles Helene (price: 0.5)</li><li>Orange Jelly (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Chestnut Coupe (price: 0.6)</li><li>Sherbet Panache (price: 0.4)</li><li>Frozen Log Plaza (price: 0.6)</li><li>Brie Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Grapefruit (price: 0.4)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Pear (price: 0.2)</li><li>Luncheon</li><li>Clams And Cocktails</li><li>Cherry Stone Clams (price: 0.5)</li><li>Cocktail Sauce (price: 0.1)</li><li>Little Neck Clams (price: 0.45)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Pineapple Paradise (price: 0.75)</li><li>Filet Of Herring (price: 0.5)</li><li>Tomato Monegasque (price: 0.75)</li><li>Stuffed Celery (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Eggs Scrambled, Grand'mere (price: 0.8)</li><li>Eggs Poached, Plaza (price: 0.55)</li><li>Omelette, Agnes Sorel (price: 0.8)</li><li>Eggs Fried Andalouse (price: 0.5)</li><li>Eggs En Cocotte Zingara (price: 0.5)</li><li>Fried Soft Shell Crabs With Bacon, Sauce Remoulade (price: 1.25)</li><li>Creamed Salmon In Bordure Florentine (price: 1.25)</li><li>Chicken Halibut Los Angeles (price: 1.1)</li><li>Filet Of Sea Bass Lady Egmont (price: 1.1)</li><li>Kingfish, Amelie (price: 1.1)</li><li>Grilled Bluefish, Jardiniere (price: 1.1)</li><li>Brook Trout à La Buena Vista (price: 1.1)</li><li>Boneless Shad And Roe Beaucaire (price: 1.25)</li><li>Supreme Of Sole Commodore (price: 0.9)</li><li>Frogs' Legs Rissolés Royer (price: 1.25)</li><li>Jersey Asparagus (price: 0.9)</li><li>Beets In Butter (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>Stewed Okra (price: 0.6)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>Lima Beans (price: 0.6)</li><li>New Peas (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>Succotash Plaza (price: 0.65)</li><li>String Beans (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Braised Lettuce Or Celery (price: 0.6)</li><li>Parsnips, Fines Herbes (price: 0.5)</li><li>Hubbard Or White Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Oyster Plant, Poulette (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Potatoes Bennet (price: 0.5)</li><li>Potatoes  Sweet (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Biron Potatoes (price: 0.45)</li><li>Voisin Potatoes (price: 0.5)</li><li>Knickerbocker Salad (price: 0.6)</li><li>Dixie Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Voisin Potatoes (price: 0.5)</li><li>Louisette Salad (price: 0.6)</li><li>Lettuce, Tomato And Cucumber Salad (price: 0.6)</li><li>Kuroki Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Endive Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Hot Tapioca Pudding (price: 0.35)</li><li>Cold Pudding Clo Clo (price: 0.35)</li><li>Chestnut Bavaroise (price: 0.4)</li><li>Souffle Rothschild (price: 0.75)</li><li>Crepes Suzette (price: 1.0)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Chestnut Coupe (price: 0.6)</li><li>Sherbet Panache (price: 0.4)</li><li>Frozen Log Plaza (price: 0.6)</li><li>Brie Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Camembert Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>King Orange (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Pear (price: 0.2)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Cocktail Sauce (price: 0.1)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Roast Beef (price: 1.0)</li><li>Baby Lamb (price: 1.5)</li><li>Broiler (price: 1.5- 3.0)</li><li>Royal Squab (price: 1.75)</li><li>Squab Chicken (price: 2.0)</li><li>Jersey Asparagus (price: 0.9)</li><li>Beets In Butter (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>Stewed Okra (price: 0.6)</li><li>Mashed Yellow Turnips (price: 0.5)</li><li>New Peas (price: 0.6)</li><li>Lima Beans (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>String Beans (price: 0.6)</li><li>Succotash Plaza (price: 0.65)</li><li>Braised Lettuce (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Hubbard Squash (price: 0.5)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Creamed Spinach (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Fried Egg Plant (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Voisin Potatoes (price: 0.5)</li><li>Hartford Salad (price: 0.6)</li><li>Lettuce, Celery And Pimentos Salad (price: 0.45)</li><li>Bonnie Brea Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.45)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Escarole Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Hot Peach Conde (price: 0.4)</li><li>Cold Jelly Roll Pudding (price: 0.35)</li><li>Bavaroise Pralinee (price: 0.4)</li><li>Souffle Nicoise (price: 0.75)</li><li>Madeira Jelly (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Chestnut Coupe (price: 0.6)</li><li>Sherbet Panache (price: 0.4)</li><li>Frozen Log Plaza (price: 0.6)</li><li>Camembert Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Stilton Cheese (price: 0.6)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>King Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Delicious Apple (price: 0.2)</li><li>Orange (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Sweet Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Plaza Potatoes (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Dixie Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Mimosa Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Lettuce, Tomato And Asparagus Tips Salad (price: 0.6)</li><li>Jeannette Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Rocheford Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Romaine Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Cocoanut Custard Pie (price: 0.35)</li><li>Diplomat Pudding (price: 0.35)</li><li>Apricot Pie (price: 0.35)</li><li>Pain De La Mecque (price: 0.25)</li><li>Strawberry Short Cake Plaza (price: 0.65)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Strawberry Mousse (price: 0.6)</li><li>Grenadine Sherbet (price: 0.4)</li><li>Parfait Praline (price: 0.6)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Honeydew Melon (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Grapefruit (price: 0.4)</li><li>Orange (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Dixie Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Lily Salad (price: 0.6)</li><li>Sunrise Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Princess Salad (price: 0.6)</li><li>Plaza Special Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Cold Macaroon Pudding (price: 0.35)</li><li>Hot Tutti Frutti Pudding (price: 0.35)</li><li>Wild Cherry Jelly (price: 0.35)</li><li>Souffle Milady (price: 0.75)</li><li>Rice Turban Cardinal (price: 0.4)</li><li>French Pastry (1) (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>Strawberry Ice (price: 0.4)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Frozen Log Plaza (price: 0.6)</li><li>Chestnut Coupe (price: 0.6)</li><li>Sherbet Panache (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Camembert Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Grapefruit (price: 0.4)</li><li>Pear (price: 0.2)</li><li>Voisin Potatoes (price: 0.5)</li><li>Plaza Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Sweet Potatoes (price: 0.5)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Dixie Potatoes (price: 0.5)</li><li>Juliette Salad (price: 0.6)</li><li>Royal Salad (price: 0.6)</li><li>Excelsior Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Lettuce Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Lemon Custard Pie (price: 0.35)</li><li>Pudding A La Reine (price: 0.35)</li><li>Rhubarb Pie (price: 0.35)</li><li>Charlotte Russe (price: 0.4)</li><li>Mocha Layer Cake (price: 0.4)</li><li>French Pastry (1) (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>Strawberry Ice (price: 0.4)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Parfait Praline (price: 0.6)</li><li>Strawberry Mousse (price: 0.6)</li><li>Grenadine Sherbet (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Camembert Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Roquefort Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Delicious Apple (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Pear (price: 0.2)</li><li>Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>King Orange (price: 0.2)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Soups</li><li>Chicken Okra (price: 0.5)</li><li>St. Germain (price: 0.4)</li><li>Tomato (price: 0.4)</li><li>Cream Of Barley Aux Perles Du Nizam (price: 0.5)</li><li>Florida With Profiterole Virginienne (price: 0.6)</li><li>Jellied Consommes    Madrilène, Chicken, Beef Or Gumbo, Cup (price: 0.4)</li><li>Fish</li><li>Boiled Kennebec Salmon, Sauce Hollandaise (price: 1.1)</li><li>Supreme Of Sole A La Plaza (price: 0.9)</li><li>Brook Trout A La Tivoli (price: 1.1)</li><li>Patty Of Frogs' Legs With Mushrooms Newburg (price: 1.25)</li><li>Sea Bass Saute Dauphine (price: 1.1)</li><li>Grilled Bluefish With Cucumber Vernon (price: 1.1)</li><li>Planked Chicken Halibut Nivernaise (price: 1.1)</li><li>Filet Of Kingfish Czarine (price: 1.1)</li><li>Soft Shell Crabs Rissolés With Almonds And Apples (price: 1.25)</li><li>Boneless Shad And Roe Meuniere A La Vichy (price: 1.25)</li><li>Baked Lobster In Shell Cardinal (1/2) (price: 1.5)</li><li>To Order</li><li>Poussin De Serre A La Washington (price: 2.25)</li><li>Mousseline Of Ham Florentine (price: 1.1)</li><li>Calf's Head A L'huile (price: 0.9)</li><li>Milk Fed Chicken Saute Bordelaise (1/2) (price: 1.75)</li><li>Turban De Cretes Et Rognons De Coq Toulousaine (price: 1.25)</li><li>Bouchee Montglas (price: 1.25)</li><li>Imperial Squab En Casserole Turner (price: 2.0)</li><li>Noisette De Pre Sale Rossini (price: 1.5)</li><li>Grilled Fresh Mushrooms On Toast With Ham Maitre D'hotel (price: 1.0)</li><li>Delmonico Steak à La Moêlle (price: 2.25)</li><li>Desserts</li><li>Hot Apple Charlotte (price: 0.4)</li><li>Cold Almond Pudding (price: 0.35)</li><li>Soufflé Saxon (price: 0.75)</li><li>Cherries Jubilee (price: 0.65)</li><li>Bavaroise Apricotine (price: 0.4)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Ice Creams  Fresh Strawberry Or Caramel (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Sherbet Panache (price: 0.4)</li><li>Chestnut Coupe (price: 0.6)</li><li>Frozen Log Plaza (price: 0.6)</li><li>Cheese  Camembert Or Brie (price: 0.45)</li><li>Swiss (price: 0.4)</li><li>Port Du Salut Or Bel Paese (price: 0.45)</li><li>Roquefort Or Gorgonzola (price: 0.5)</li><li>Stilton (price: 0.6)</li><li>Liederkranz (price: 0.45)</li><li>English Cheddar (price: 0.5)</li><li>Fruits  Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>King Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Orange, Delicious Apple Or Pear (price: 0.2)</li><li>Grapes (price: 0.5)</li><li>Chicken Halibut A La Venitienne (price: 1.1)</li><li>Boston Rock Cod Saute Lyonnaise (price: 0.9)</li><li>Filet Of Kingfish, Laperouse (price: 1.1)</li><li>Supreme Of Sole Vladimir (price: 0.9)</li><li>Brook Trout A La Beaucaire (price: 1.1)</li><li>Sea Bass Saute Cleopatra (price: 1.1)</li><li>Crab Meat Cakes A La Hoyt (price: 1.25)</li><li>Grenadin Of Salmon Petit Trianon (price: 1.25)</li><li>Boneless Shad And Roe, Sauce Raifort, Bacon, Potatoes Bataille (price: 1.25)</li><li>Frogs' Legs Bercy With Rice Pilaw (price: 1.25)</li><li>Cold Chicken Lobster, Sauce Vincent (price: 1.5)</li><li>Prime Rib Of Beef With Potatoes Macaire And Stuffed Tomato (price: 1.25)</li><li>Smothered Squab Chicken A La Cote D'azur (1/2) (price: 1.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Plaza Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Louisette Salad (price: 0.6)</li><li>Lettuce, Tomato And Cucumber Salad (price: 0.6)</li><li>Kuroki Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Fruit Salad (price: 0.6)</li><li>Escarole Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Farina Pudding, Peach Sauce (price: 0.35)</li><li>Strawberry Meringue Pie (price: 0.35)</li><li>Pear Pie (price: 0.35)</li><li>Cheese Cake (price: 0.4)</li><li>Savarin Curacao (price: 0.25)</li><li>Rice Pudding (price: 0.35)</li><li>Apple Pie (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Grenadine Sherbet (price: 0.4)</li><li>Strawberry Mousse (price: 0.6)</li><li>Parfait Praline (price: 0.6)</li><li>Brie (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Camembert Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Cantaloupe (price: 0.6)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>King Orange (price: 0.2)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Salami (price: 0.5)</li><li>Chicken Okra Soup (price: 0.5)</li><li>St. Germain Soup (price: 0.4)</li><li>Tomato Soup (price: 0.4)</li><li>Stuffed Brook Trout Polonaise (price: 1.1)</li><li>Boiled Live Codfish, Egg Sauce (price: 0.9)</li><li>Planked Boneless Shad A La Hoffman (price: 1.25)</li><li>Kingfish Saute Florida (price: 1.1)</li><li>Soft Shell Crabs, Belle Meuniere (price: 1.25)</li><li>Supreme Of Sole Amelie (price: 0.9)</li><li>Lobster Excelsior (1/2) (price: 1.5)</li><li>Grilled Bluefish, Tomato Relish (price: 1.1)</li><li>Filet Of Sea Bass Breval (price: 1.1)</li><li>Patty Of Frogs' Legs Maryland (price: 1.25)</li><li>Darne Of Salmon, Imperatrice (price: 1.25)</li><li>Halibut Steak Meuniere A L'orange (price: 1.1)</li><li>Roast Rack And Saddle Of Baby Lamb, Boulangere, Creamed Asparagus (price: 1.75)</li><li>Glazed Sugar Cured Ham With Candied Sweet Potatoes, Whole Spinach (price: 1.1)</li><li>Frincandeau Of Milk Veal, National (price: 1.25)</li><li>Petite Tournedo Saute Samaritaine (price: 1.75)</li><li>Grilled Kernel Of Sweetbread With Heart Of Artichoke Da Gama (price: 1.5)</li><li>Fried Squab Chicken A La Louisiana (1/2) (price: 1.5)</li><li>Quenelle Of Capon, Sauce Archiduc With Puree Of Peas (price: 1.25)</li><li>Poussin De Serre A La Souvaroff (price: 2.25)</li><li>Boneless Royal Squab A La Washington (price: 2.0)</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Plaza Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Knickerbocker Salad (price: 0.6)</li><li>Lettuce, Orange And Grapefruit Salad (price: 0.6)</li><li>Blackstone Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Escarole Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Mixed Green Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Hot Semoulina Pudding (price: 0.35)</li><li>Cold Cabinet Pudding (price: 0.35)</li><li>Fresh Pear Bourdaloue (price: 0.4)</li><li>Orange Souffle (price: 0.75)</li><li>Lemon Jelly (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Apple Pie (price: 0.35)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Sherbet Panache (price: 0.4)</li><li>Chestnut Coupe (price: 0.6)</li><li>Frozen Log Plaza (price: 0.6)</li><li>Camembert Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Brie Cheese (price: 0.45)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Honeydew Melon (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Grapefruit (price: 0.4)</li><li>Grapes (price: 0.5)</li><li>King Orange (price: 0.2)</li><li>Orange (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>Pear (price: 0.2)</li><li>Cocktail Sauce (price: 0.1)</li><li>Cherry Stone Clams (price: 0.5)</li><li>Little Neck Clams (price: 0.45)</li><li>Crab Flake Cocktail (price: 0.75)</li><li>Sea Food Cocktail Plaza (price: 1.0)</li><li>Clam Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Tomato Juice Cocktail (price: 0.35)</li><li>Shrimp Cocktail (price: 0.75)</li><li>Hors D'oeuvres</li><li>Pineapple Paradise (price: 0.75)</li><li>Orange, Grapefruit, Melon Or Fruit Supreme (price: 0.65)</li><li>Fillet Of Herring (price: 0.5)</li><li>Petit Artichoke Mariniere (price: 0.5)</li><li>Smoked Salmon (price: 0.75)</li><li>Tomato Monagasque (price: 0.75)</li><li>Plaza Appetizer (price: 1.25)</li><li>Anchovy Canape (price: 0.6)</li><li>Stuffed Celery (price: 0.5)</li><li>Hors D'oeuvre Variés</li><li>Salami (price: 0.5)</li><li>Antipasto (price: 0.5)</li><li>Westphalia Ham (price: 1.25)</li><li>Maquereau Au Vin Blanc (price: 0.5)</li><li>Chicken Okra Soup (price: 0.5)</li><li>Tomato Soup (price: 0.4)</li><li>St. Germain (price: 0.4)</li><li>Julienne Mongole (price: 0.5)</li><li>Essence De Tomate With Egg Filets (price: 0.4)</li><li>Jellied Comsommes   Madrilene, Chicken, Beef Or Gumbo (price: 0.4)</li><li>Scrambled Eggs With Mushrooms (price: 0.8)</li><li>Eggs En Cocotte Café Anglais (price: 0.5)</li><li>Omlette Creole (price: 0.8)</li><li>Eggs Poached, Comtesse (price: 0.55)</li><li>Eggs Mollet à L'oseille (price: 0.5)</li><li>Boston Scrod Vendome (price: 0.9)</li><li>Creamed Finnan Haddie A La Kitchen (price: 0.9)</li><li>Brook Trout A La Vatel (price: 1.1)</li><li>Boneless Shad And Roe Saute Mascotte (price: 1.25)</li><li>Filet Of Sole Normande (price: 0.9)</li><li>Soft Shell Crabs Under Glass Maryland (price: 1.25)</li><li>Bluefish Epicurienne (price: 1.1)</li><li>Fried Frogs' Legs With Bacon, Sauce Orly (price: 1.25)</li><li>Supreme Of Sea Bass Florentine (price: 1.1)</li><li>Chicken Halibut Marseillaise (price: 1.1)</li><li>Cold Chicken Lobster Remoulade (price: 1.5)</li><li>Cold Salmon A L'espagnole (price: 1.25)</li><li>Prime Rib Of Beef With Yorkshire Pudding And New Succotash (price: 1.25)</li><li>Grilled Milk Veal Chop, Continental (price: 1.25)</li><li>Squab Chicken Pot Pie à La Plaza (price: 1.5)</li><li>Deerfoot Sausages Gastronome (price: 0.9)</li><li>Noisette Of Spring Lamb Salvandi (price: 1.5)</li><li>Assorted Fresh Vegetables In Bordure Gratinée (price: 1.25)</li><li>Macaroni Caruso (price: 0.9)</li><li>Gnocchi Romaine (price: 0.5)</li><li>Spaghetti A La Milanaise (price: 0.75)</li><li>Pilaw Of Duck Livers Bordelaise (price: 0.9)</li><li>Royal Squan En Casserole Beaulieu (price: 2.0)</li><li>Supreme Of Guinea Hen, Metternick (price: 1.75)</li><li>Arroz Con Pollo Catalane (price: 1.75)</li><li>Baby Lamb Steak Saute Nicoise (price: 2.0)</li><li>Vol Au Vent Of Calf's Brains Poulette (price: 1.0)</li><li>Poussin De Serre A La Washington (price: 2.25)</li><li>Small Steak En Casserole Nivernaise (price: 2.25)</li><li>Fresh Mushrooms On Toast With Bacon, Maitre D'hotel (price: 1.0)</li><li>Beets In Butter (price: 0.5)</li><li>Cauliflower (price: 0.7)</li><li>Jersey Asparagus (price: 0.9)</li><li>Lima Beans (price: 0.6)</li><li>String Beans (price: 0.6)</li><li>Carrots Vichy (price: 0.5)</li><li>Succotash Plaza (price: 0.65)</li><li>New Peas (price: 0.6)</li><li>Stewed Okra (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Artichoke (price: 0.6)</li><li>Braised Lettuce (price: 0.6)</li><li>Braised Celery (price: 0.6)</li><li>Creamed Spinach (price: 0.6)</li><li>Stewed Fresh Tomatoes (price: 0.5)</li><li>White Squash (price: 0.5)</li><li>Hubbard Squash (price: 0.5)</li><li>Fried Egg Plant (price: 0.5)</li><li>Mashed Yellow Turnips (price: 0.6)</li><li>Small Onions In Cream (price: 0.6)</li><li>Potatoes</li><li>Biron Potatoes (price: 0.45)</li><li>Boiled Bermuda Potatoes (price: 0.3)</li><li>Plaza Potatoes (price: 0.5)</li><li>Sweet Potatoes (price: 0.5)</li><li>Dixie Potatoes (price: 0.5)</li><li>Voisin Potatoes (price: 0.5)</li><li>Lettuce, Celery And Pimentos Salad (price: 0.6)</li><li>Hartford Salad (price: 0.6)</li><li>Bonnie Brae Salad (price: 0.6)</li><li>Rocheford Salad (price: 0.6)</li><li>Fruit Salad (price: 0.6)</li><li>Alligator Pear Salad (price: 0.7)</li><li>Mixed Green Salad (price: 0.45)</li><li>Escarole Salad (price: 0.45)</li><li>Endive Salad (price: 0.45)</li><li>Chicory Salad (price: 0.45)</li><li>Romaine Salad (price: 0.45)</li><li>Tomato Salad (price: 0.45)</li><li>Lettuce Salad (price: 0.45)</li><li>Boston Cream Pie (price: 0.35)</li><li>Bread And Butter Pudding (price: 0.35)</li><li>Peach Pie (price: 0.35)</li><li>Old Fashioned Strawberry Short Cake (price: 0.65)</li><li>Black Cherry Tart (price: 0.4)</li><li>Apple Pie (price: 0.35)</li><li>Rice Pudding (price: 0.35)</li><li>French Pastry (1) (price: 0.25)</li><li>Fresh Strawberry Ice Cream (price: 0.45)</li><li>Strawberry Ice (price: 0.4)</li><li>Caramel Ice Cream (price: 0.45)</li><li>Lemon Sherbet (price: 0.4)</li><li>Vanilla Mousse, Hot Chocolate Sauce (price: 0.6)</li><li>Parfait Tosca (price: 0.6)</li><li>Camembert Cheese (price: 0.45)</li><li>Swiss Cheese (price: 0.4)</li><li>Port Du Salut Cheese (price: 0.45)</li><li>Brie Cheese (price: 0.45)</li><li>Bel Paese Cheese (price: 0.45)</li><li>Roquefort Cheese (price: 0.5)</li><li>Gorgonzola Cheese (price: 0.5)</li><li>Stilton Cheese (price: 0.6)</li><li>English Cheddar Cheese (price: 0.5)</li><li>Liederkranz Cheese (price: 0.45)</li><li>Honeydew Melon (price: 0.6)</li><li>Strawberries (price: 0.6)</li><li>Cantaloupe (price: 0.6)</li><li>Grapes (price: 0.5)</li><li>Orange (price: 0.2)</li><li>Grapefruit (price: 0.4)</li><li>Pear (price: 0.2)</li><li>Delicious Apple (price: 0.2)</li><li>King Orange (price: 0.2)</li></ul>"
					}
				],
				"seeAlso": []
			}
		]
	},
	{
		"type": "web",
		"url": "http://menus.nypl.org/menu_pages/55739",
		"items": [
			{
				"itemType": "document",
				"title": "Legal Sea Foods menu",
				"creators": [],
				"date": "1998-01-26",
				"callNumber": "1998-0005_wotm",
				"extra": "genre: Restaurant menu\nRestaurant Location: Legal Sea Foods\nPhysical Description: 37.5x23 folded; 37.5x46cm open",
				"libraryCatalog": "New York Public Library Menu Collection",
				"url": "http://menus.nypl.org/menu_pages/55736",
				"attachments": [],
				"tags": [],
				"notes": [
					{
						"note": "<h1>Dishes</h1><ul><li>Cherrystone Clams. Six On The Half Shell...With Any Of The Littleneck Garnishes (price: 6.95)</li><li>Oysters. Six On The Half Shell...Several Varieties Available...With Your Choice Of Lemon And Pepper, Champagne Migonette, Grated Horseradish, Green Tabasco Or Cocktail Sauce (price: 8.95)</li><li>Littleneck Clams. Eight On The Half Shell...With Your Choice Of Lemon And Horseradish, Lime And Chile Salsa, Green Tabasco Or Cocktail Sauce (price: 7.95)</li><li>Steamers With Drawn Butter (price: 9.95)</li><li>Mussels With Garlic And White Wine (price: 8.95)</li><li>Mussels With Fragrant Seafood Sauce (price: 8.95)</li><li>Baltimore Peel N' Eat Shrimp (price: 9.95)</li><li>Legal's New England Clam Chowder. Served At The Past Five Presidential Inaugurations And Boston's Fenway Park, Too! (price: 3.5- 4.25)</li><li>Rhode Island Red Clam Chowder (price: 3.5- 4.25)</li><li>Fish Chowder (price: 3.95- 4.95)</li><li>Fried Smelts. Butterflied And Boneless (price: 4.95)</li><li>C Bar Coconut Shrimp. With Ginger Orange Sauce (price: 8.95)</li><li>Shrimp Wontons. Steamed Or Fried (price: 6.95)</li><li>Fried Oysters. With Cassia Salt And Seaweed Salad (price: 9.95)</li><li>Legal's Smoked Bluefish Pate. The Original...Rolled In A Walnut And Herb Crust (price: 5.95)</li><li>Onion Strings. Thinly Sliced, Battered And Fried (price: 3.95)</li><li>Kung Pao Chicken Wings. Spicy...Served With Blue Cheese (price: 6.95)</li><li>Rhode Island Fried Calamari. Hot Peppers & Garlic (price: 7.95)</li><li>Mussels Au Gratin. With Garlic (price: 5.95- 9.95)</li><li>Oysters Legal. Baked With Spinach, Cheese & Crumbs (price: 9.95)</li><li>Jonah Crab Claws. On Ice (price: 9.95)</li><li>Smoked Salmon. Thinly Sliced...Pumpernickle Toast (price: 11.95)</li><li>Maryland Crab Cake. With Apple And Mesclun Salad, Cider Mustard Vinaigrette (price: 10.95)</li><li>Clam Fritters. With Bacon...Roasted Red Pepper Sauce. Great With Chowder! (price: 5.95)</li><li>Shrimp Cocktail (price: 8.95)</li><li>House Salad. A Mix Of Romaine And Mesclun With Red Onions, Tomatoes And Cucumbers...Choice Of Dressing (price: 3.5)</li><li>Caesar Salad (price: 4.95)</li><li>Winter Salad. Endive, Pear And Maytag Blue Cheese With Toasted Walnuts (price: 5.95)</li><li>Seafood Antipasto. Grilled Shrimp, Clams, Mussels And Calamari...With Greens, Provolone Cheese And Olives...Tomato Vinaigrette (price: 14.95)</li><li>Warm Marinated And Grilled Calamari. With Grilled Onions And White Beans (price: 8.95)</li><li>Steamed Vegetables (price: 3.95)</li><li>Steamed Broccoli...With Or Without Cheese (price: 3.95)</li><li>Mashed Potatoes (price: 2.25)</li><li>French Fries (price: 2.25)</li><li>Sugar Snap Peas...Oyster Sauce (price: 3.95)</li><li>Baked Idaho Potato (price: 2.25)</li><li>Rice Pilaf With Orzo (price: 2.25)</li><li>Vegetable Of The Day (price: 2.25)</li><li>Onion Strings (price: 2.25)</li><li>Jasmine Rice (price: 2.25)</li><li>Vegetable Slaw (price: 2.25)</li><li>Lobster Bake. Steamed 1 1.25 Lb Lobster, Steamers, Mussels, Corn On Th Cob, And A Cup Of Clam Chowder...With One Side (price: 34.95)</li><li>Steamed Lobster (price: 19.95- 39.95)</li><li>Baked Stuffed Lobster With Buttery Crabmeat Stuffing (price: 22.95- 44.95)</li><li>Salmon Fillet (price: 18.95)</li><li>Tuna. Prepared Medium Rare (price: 19.95)</li><li>Assorted Dinner. Shrimp, Scallops & Three Cuts Of Fish (price: 19.95)</li><li>Rainbow Trout (price: 12.95)</li><li>Sirloin Steak 12 Oz. (price: 18.95)</li><li>Bluefish. Also Available Baked With Mustard Sauce (price: 11.95)</li><li>Arctic Char (price: 19.95)</li><li>Filet Mignon 8 Oz. (price: 22.95)</li><li>Haddock Loin (price: 16.95)</li><li>Skewered Shrimp (price: 18.95)</li><li>Chicken Breast (price: 13.95)</li><li>Mahi Mahi (price: 17.95)</li><li>Sea Scallops (price: 19.95)</li><li>Homemade Seafood Sausage. Made With Shrimp, Salmon And White Fish...Served With Savoy Cabbage And Mashed Potatoes. (price: 16.95)</li><li>Portuguese Fisherman's Stew. Cod, Mussels, Clams And Chorizo In A Saffron Tomato Broth...Garlick Toasts (price: 14.95)</li><li>Swordfish. Grilled And Served With Sauteed Cherry Tomatoes, Rice Pilaf And Cold Cucumber Sauce (price: 22.95)</li><li>\"Scampi\". Shrimp And Garlic With White Wine, Tomatoes And Mushrooms Over Linguine (price: 17.95)</li><li>Classic Boston Scrod. Baked With Buttery Bread Crumbs...With Seasonal Vegetables And Choice Of One Side (price: 14.95)</li><li>C Bar Brazilian Red Snapper Fillet. Cooked With Coconut And Cilantro (price: 18.95)</li><li>Seafood Casserole. Scallops, Shrimp, Lobster And White Fish...\"Au Gratin\"...Choice Of One Side (price: 21.95)</li><li>Flounder Fillet. Plain, Simple And Delicious...Broiled Or Baked With Buttery Bread Crumbs...With Seasonal Vegetables And Choice Of One Side (price: 17.95)</li><li>Linguini With White Clam Sauce. Lots Of Whole And Chopped Clams, Garlic, Parmesan, Herbs And Butter (price: 15.95)</li><li>Jasmine Special. Shrimp And Broccoli Steamed Over Jasmine Rice...Melted Monterey Jack Cheese (price: 15.95)</li><li>Salmon Fillet With Chinese Fruit Sauce Or Fragrant Seafood Sauce. With Sugar Snap Peas And Jasmine Rice (price: 15.95)</li><li>Baked Stuffed Shrimp. Baked With A Buttery Crabmeat Stuffing...Choice Of One Side (price: 17.95)</li><li>Vegetarian Curried Noodles With Small Shrimp. Chinese Egg Noodles Stir Fried With Leeks, Bok Choi, Eggplant And Tofu (price: 13.95)</li><li>Vegetarian Curried Noodles. Chinese Egg Noodles Stir Fried With Leeks, Bok Choi, Eggplant And Tofu (price: 10.95)</li><li>Spicy Shandong Baked Bluefish. With Sugar Snap Peas And Jasmine Rice (price: 12.95)</li><li>\"Smoke Roasted\" Salmon Cold Plate. Cooked And Chilled Smoked Salmon From Maine With Roasted Red Pepper Sauce, Capers, Coucous Salad, Pickled Red Onions, And Hard Boiled Egg (price: 13.95)</li><li>Haddock (price: 14.95)</li><li>Calamari. Regular Or Spicy (price: 11.95)</li><li>Scrod (price: 14.95)</li><li>Sea S Callops (price: 15.95)</li><li>Fish & Chips. Regular Or Spicy (price: 11.95)</li><li>Fisherman's Platter. Shrimp, Clam Fritters, Scallops, Clams & Haddock (price: 19.95)</li><li>Smelts (price: 9.95)</li><li>Maine Clams (price: 15.95)</li><li>Sole Flakes (price: 15.95)</li><li>Arctic Char (price: 14.95)</li><li>Rainbow Trout (price: 11.95)</li><li>Skewered Shrimp (price: 13.95)</li><li>Bluefish (price: 7.95)</li><li>Shrimp Wontons. Fried Or Steamed...With Snap Peas And Jasmine Rice (price: 8.95)</li><li>\"Smoke Roasted\" Salmon Cold Plate. Cooked And Chilled Smoked Salmon From Maine With Roasted Red Pepper Sauce, Capers, Coucous Salad, Pickled Red Onions, And Hard Boiled Egg (price: 13.95)</li><li>Trout Provencal. Sauteed With Diced Tomatoes, Mushrooms And Garlic...With Rice (price: 11.95)</li><li>Caesar Salad With Cajun Shrimp. Fresh Romaine Tossed With Ceasar Drissing, Croutons (price: 13.95)</li><li>\"Scampi\" Shrimp And Garlic With White Wine, Tomatoes And Mushrooms Over Linguine (price: 11.95)</li><li>Gumbo. Hearty & Spicy...Shrimp, Scallps, Pollock...Cajun Andouille And Tasso...Jasmine Rice (price: 10.95)</li><li>Jasmine Special. Shrimp And Broccoli Steamed Over Jasmine Rice...Melted Monterey Jack Cheese (price: 10.95)</li><li>Mussels Fra'diavolo. Mussels In A Zesty Tomato Sauce Over Linguine (price: 6.95)</li><li>Classic Boston Scrod. Baked With Buttery Bread Crumbs...With Choice Of One Side (price: 9.95)</li><li>Fish & Chips. Regular Or Spicy (price: 8.95)</li><li>Lobster Roll. Fresh Lobster Meat From A Stemed 1 1.25 Lb. Lobster...Cold, Creamy Lobster Salad In A Toasted Bun...With French Fries (price: 16.95)</li><li>Crab Roll. Maine Crab Salad In A Warm Buttery New England Bun...With Fries (price: 10.95)</li><li>Clam Fritters & Chowder. Served With Roasted Red Pepper Sauce...A Great Combination (price: 7.95)</li><li>Grilled Chicken Sandwich. Topped With Melted Cheese And Crisp Bacon (price: 8.95)</li><li>Smoked Salmon Sandwich. Fresh Egg Bread...Mustard Sauce, Sliced Tomatoes, Onions And Capers (price: 8.95)</li><li>Po'boy Sandwich. Fried Oysters, Lettuce, Tomato & Spicy Mayo (price: 9.95)</li><li>Tuna Burger. Freshly Ground Tuna, Chili Paste, Crumbs And Spices (price: 9.95)</li></ul>"
					}
				],
				"seeAlso": []
			}
		]
	},
	{
		"type": "web",
		"url": "http://menus.nypl.org/menu_pages/35364/explore",
		"items": [
			{
				"itemType": "document",
				"title": "American Asiatic Association menu",
				"creators": [],
				"date": "1906-02-03",
				"abstractNote": "WINES INCLUDED; EVENT WAS 1P.M. MENU MORE APPROPIATE FOR LUNCH THAN BREAKFAST;",
				"callNumber": "1906-126",
				"extra": "genre: Restaurant menu\nRestaurant Location: American Asiatic Association\nPlace: MERCHANTS CLUB,NY\nPhysical Description: BOOKLET; ILLUS; 4.5X7;",
				"libraryCatalog": "New York Public Library Menu Collection",
				"url": "http://menus.nypl.org/menu_pages/35366",
				"attachments": [],
				"tags": [],
				"notes": [
					{
						"note": "<h1>Dishes</h1><ul><li>Oysters</li><li>Cotuits</li><li>Clear Green Turtle Soup In Cups</li><li>Amontillado</li><li>Olives</li><li>Celery</li><li>Radishes</li><li>Filet Of Sea Bass, White Wine Sauce</li><li>Cucumber Salad</li><li>Carbonnieux 1889</li><li>Parisienne Potatoes</li><li>Noisettes Of Lamb, Bonnefoy</li><li>G. H. Mumm's Selected Brut</li><li>Fresh Artichokes, Sauce Hollandaise</li><li>Roast Chicken</li><li>Roast Squab</li><li>Salad Romaine</li><li>Ices</li><li>Assorted Cakes</li><li>Cheese</li><li>Fruit</li><li>Coffee</li></ul>"
					}
				],
				"seeAlso": []
			}
		]
	},
	{
		"type": "web",
		"url": "http://menus.nypl.org/dishes/9512",
		"items": "multiple"
	},
	{
		"type": "web",
		"url": "http://menus.nypl.org/search?utf8=%E2%9C%93&query=aspic",
		"defer": true,
		"items": "multiple"
	}
]
/** END TEST CASES **/
